<?php
/**
 * ===========================================
 * FLOWBOT DCI - MAIN APPLICATION CLASS
 * ===========================================
 * v2.2: Added security hardening (SEC-005, SEC-006, SEC-007, SEC-013)
 */

declare(strict_types=1);

namespace FlowbotDCI\Core;

use FlowbotDCI\Services\UrlProcessor;
use FlowbotDCI\Services\ProgressTracker;

class Application
{
    private array $config;
    private ?Database $database = null;
    private ?UrlProcessor $processor = null;
    private ?ProgressTracker $tracker = null;
    private string $processId = '';

    // SEC-007: Rate limiting settings
    private const RATE_LIMIT_REQUESTS = 60;   // Max requests per window
    private const RATE_LIMIT_WINDOW = 60;     // Window in seconds

    public function __construct(array $config)
    {
        $this->config = $config;

        // Set timezone first
        $timezone = $config['app']['timezone'] ?? 'America/Sao_Paulo';
        if (!empty($timezone)) {
            date_default_timezone_set($timezone);
        }

        // Error reporting based on debug mode
        $debug = $config['app']['debug'] ?? false;
        if ($debug) {
            error_reporting(E_ALL);
            ini_set('display_errors', '1');
        } else {
            error_reporting(0);
            ini_set('display_errors', '0');
        }

        // No time limit for processing
        set_time_limit(0);

        // Initialize database connection
        try {
            $this->database = new Database($config['database']);
        } catch (\Exception $e) {
            throw new \Exception("Database connection failed: " . $e->getMessage());
        }

        // SEC-013: Set security headers early
        $this->setSecurityHeaders();
    }

    /**
     * SEC-013: Set security headers
     */
    private function setSecurityHeaders(): void
    {
        // Prevent clickjacking
        header('X-Frame-Options: SAMEORIGIN');
        // XSS protection (legacy, but doesn't hurt)
        header('X-XSS-Protection: 1; mode=block');
        // Prevent MIME sniffing
        header('X-Content-Type-Options: nosniff');
        // Referrer policy
        header('Referrer-Policy: strict-origin-when-cross-origin');
        // Content Security Policy
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:; font-src 'self';");
    }

    /**
     * SEC-005: Sanitize input string
     */
    private function sanitizeInput(string $input): string
    {
        // Remove null bytes
        $input = str_replace("\0", '', $input);
        // Trim whitespace
        $input = trim($input);
        // Convert special characters to HTML entities
        return htmlspecialchars($input, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }

    /**
     * SEC-005: Sanitize URLs input (allows URLs but removes dangerous content)
     */
    private function sanitizeUrlsInput(string $input): string
    {
        // Remove null bytes
        $input = str_replace("\0", '', $input);
        // Remove any script tags
        $input = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $input);
        // Trim whitespace
        return trim($input);
    }

    /**
     * SEC-007: Check rate limit
     * Returns true if request should be allowed, false if rate limited
     */
    private function checkRateLimit(): bool
    {
        $ip = $this->getClientIp();
        $rateLimitFile = $this->config['paths']['temp'] . '/rate_limit_' . md5($ip) . '.json';

        $now = time();
        $requests = [];

        // Load existing rate limit data
        if (file_exists($rateLimitFile)) {
            $data = json_decode(file_get_contents($rateLimitFile), true);
            if (is_array($data)) {
                // Filter out old requests
                $requests = array_filter($data, function($timestamp) use ($now) {
                    return $timestamp > ($now - self::RATE_LIMIT_WINDOW);
                });
            }
        }

        // Check if rate limit exceeded
        if (count($requests) >= self::RATE_LIMIT_REQUESTS) {
            return false;
        }

        // Add current request
        $requests[] = $now;

        // Save updated rate limit data
        file_put_contents($rateLimitFile, json_encode(array_values($requests)));

        return true;
    }

    /**
     * Get client IP address
     */
    private function getClientIp(): string
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';

        // Check for proxy headers (but validate them)
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        }

        // Validate IP format
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            $ip = '127.0.0.1';
        }

        return $ip;
    }

    /**
     * SEC-006: Generate CSRF token
     */
    private function generateCsrfToken(): string
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }

        return $_SESSION['csrf_token'];
    }

    /**
     * SEC-006: Validate CSRF token
     */
    private function validateCsrfToken(): bool
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        $token = $_POST['csrf_token'] ?? '';

        if (empty($_SESSION['csrf_token']) || empty($token)) {
            return false;
        }

        return hash_equals($_SESSION['csrf_token'], $token);
    }

    /**
     * Send rate limit exceeded response
     */
    private function sendRateLimitResponse(): void
    {
        http_response_code(429);
        header('Content-Type: application/json');
        header('Retry-After: ' . self::RATE_LIMIT_WINDOW);
        echo json_encode([
            'success' => false,
            'error' => 'Rate limit exceeded. Please try again later.',
        ]);
        exit;
    }

    /**
     * Get or generate process ID
     */
    public function getProcessId(): string
    {
        if (!empty($_REQUEST['process_id'])) {
            $id = preg_replace('/[^a-zA-Z0-9_\-\.]/', '', $_REQUEST['process_id']);
            if (!empty($id)) {
                return $id;
            }
        }
        return uniqid('batch_', true);
    }

    /**
     * Initialize the application
     */
    public function initialize(): self
    {
        $this->processId = $this->getProcessId();
        $this->tracker = new ProgressTracker($this->config, $this->processId);
        $this->processor = new UrlProcessor($this->config, $this->database, $this->tracker);

        // Clean old temp folders
        $this->cleanOldFolders();

        return $this;
    }

    /**
     * Handle incoming request
     * Enhanced with page routing and API endpoints
     * v2.2: Added security hardening (rate limiting, CSRF, input sanitization)
     */
    public function handleRequest(): void
    {
        // SEC-007: Check rate limit first
        if (!$this->checkRateLimit()) {
            $this->sendRateLimitResponse();
            return;
        }

        // Get request method and page safely
        $requestMethod = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $page = $this->sanitizeInput($_GET['page'] ?? 'dashboard');

        // POST with URLs - Initialize new process
        if ($requestMethod === 'POST' && isset($_POST['web_url']) && !empty(trim($_POST['web_url']))) {
            // SEC-006: Validate CSRF token on POST requests
            if (!$this->validateCsrfToken()) {
                http_response_code(403);
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'error' => 'Invalid or missing CSRF token.',
                ]);
                return;
            }

            // SEC-005: Sanitize URL input (allows URLs but removes dangerous content)
            $sanitizedUrls = $this->sanitizeUrlsInput($_POST['web_url']);
            $this->initializeProcess($sanitizedUrls);
            return;
        }

        // Check if progress file exists
        if (!$this->tracker || !$this->tracker->hasProgress()) {
            $this->showForm();
            return;
        }

        // Handle API requests
        if ($page === 'api') {
            $this->handleApi();
            return;
        }

        // Page routing
        switch ($page) {
            case 'logs':
                $this->showLogsPage();
                break;
            case 'domains':
                $this->showDomainsPage();
                break;
            default:
                // Dashboard - process batch and show progress
                $this->processBatch();
        }
    }

    /**
     * Handle API requests for AJAX calls
     * v2.2: Added input sanitization for all parameters
     */
    private function handleApi(): void
    {
        header('Content-Type: application/json');
        header('Cache-Control: no-cache, must-revalidate');

        // SEC-005: Sanitize action parameter
        $action = $this->sanitizeInput($_GET['action'] ?? '');

        switch ($action) {
            case 'progress':
                echo json_encode([
                    'success' => true,
                    'data' => $this->tracker->loadProgress(),
                ]);
                break;

            case 'domains':
                // SEC-005: Sanitize sort parameters
                $sortBy = $this->sanitizeInput($_GET['sort'] ?? 'total');
                $direction = $this->sanitizeInput($_GET['dir'] ?? 'desc');
                // Validate allowed values
                $allowedSortBy = ['total', 'success', 'error', 'ignored', 'rate', 'domain', 'avg_time'];
                $allowedDir = ['asc', 'desc'];
                $sortBy = in_array($sortBy, $allowedSortBy) ? $sortBy : 'total';
                $direction = in_array($direction, $allowedDir) ? $direction : 'desc';
                echo json_encode([
                    'success' => true,
                    'data' => $this->tracker->getDomainStatsSorted($sortBy, $direction),
                ]);
                break;

            case 'logs':
                // SEC-005: Sanitize all filter parameters
                $filters = [
                    'status' => $this->sanitizeInput($_GET['status'] ?? ''),
                    'domain' => $this->sanitizeInput($_GET['domain'] ?? ''),
                    'error_type' => $this->sanitizeInput($_GET['error_type'] ?? ''),
                    'search' => $this->sanitizeInput($_GET['search'] ?? ''),
                ];
                $page = max(1, (int)($_GET['p'] ?? 1));
                $perPage = min(100, max(10, (int)($_GET['per_page'] ?? 50))); // Limit to 10-100

                echo json_encode([
                    'success' => true,
                    'data' => $this->tracker->getFilteredLogs($filters, $page, $perPage),
                ]);
                break;

            case 'log_domains':
                echo json_encode([
                    'success' => true,
                    'data' => $this->tracker->getLogDomains(),
                ]);
                break;

            case 'export':
                $this->handleExport();
                break;

            default:
                echo json_encode([
                    'success' => false,
                    'error' => 'Unknown action',
                ]);
        }

        exit;
    }

    /**
     * Handle data export requests
     * v2.2: Added input validation and sanitization
     */
    private function handleExport(): void
    {
        // SEC-005: Sanitize and validate export parameters
        $type = $this->sanitizeInput($_GET['type'] ?? 'logs');
        $format = $this->sanitizeInput($_GET['format'] ?? 'json');

        // Validate allowed values
        $allowedTypes = ['logs', 'domains'];
        $allowedFormats = ['json', 'csv'];
        $type = in_array($type, $allowedTypes) ? $type : 'logs';
        $format = in_array($format, $allowedFormats) ? $format : 'json';

        switch ($type) {
            case 'domains':
                $data = $this->tracker->getDomainStatsSorted();
                $filename = "domains_{$this->processId}";
                break;
            case 'logs':
            default:
                $logsData = $this->tracker->getFilteredLogs([], 1, 10000);
                $data = $logsData['logs'];
                $filename = "logs_{$this->processId}";
        }

        if ($format === 'csv') {
            header('Content-Type: text/csv');
            header("Content-Disposition: attachment; filename=\"{$filename}.csv\"");

            $output = fopen('php://output', 'w');
            if (!empty($data)) {
                // Get headers from first item
                $first = is_array($data) ? reset($data) : $data;
                if (is_array($first)) {
                    fputcsv($output, array_keys($first));
                    foreach ($data as $row) {
                        // Flatten arrays for CSV
                        $flatRow = array_map(function($val) {
                            return is_array($val) ? json_encode($val) : $val;
                        }, $row);
                        fputcsv($output, $flatRow);
                    }
                }
            }
            fclose($output);
        } else {
            header('Content-Type: application/json');
            header("Content-Disposition: attachment; filename=\"{$filename}.json\"");
            echo json_encode($data, JSON_PRETTY_PRINT);
        }

        exit;
    }

    /**
     * Show the domains statistics page
     */
    private function showDomainsPage(): void
    {
        $viewPath = $this->config['paths']['views'] . '/domains.php';

        if (!file_exists($viewPath)) {
            throw new \Exception("View file not found: {$viewPath}");
        }

        $data = $this->tracker->loadProgress();
        $domainStats = $this->tracker->getDomainStatsSorted();
        $processId = $this->processId;
        $currentPage = 'domains';

        include $viewPath;
        exit;
    }

    /**
     * Show the logs page
     */
    private function showLogsPage(): void
    {
        $viewPath = $this->config['paths']['views'] . '/logs.php';

        if (!file_exists($viewPath)) {
            throw new \Exception("View file not found: {$viewPath}");
        }

        $data = $this->tracker->loadProgress();
        $logsData = $this->tracker->getFilteredLogs([], 1, 50);
        $logDomains = $this->tracker->getLogDomains();
        $processId = $this->processId;
        $currentPage = 'logs';

        include $viewPath;
        exit;
    }

    /**
     * Initialize new processing session
     */
    private function initializeProcess(string $rawUrls): void
    {
        $links = $this->processor->extractUrls($rawUrls);
        $this->tracker->initialize($links);

        header('Content-Type: text/html; charset=UTF-8');
        echo "<!DOCTYPE html><html><head><meta charset='UTF-8'>";
        echo "<title>Process Started</title>";
        echo "<style>body{font-family:Arial,sans-serif;background:#2c3e50;color:#fff;display:flex;justify-content:center;align-items:center;min-height:100vh;margin:0;}";
        echo ".box{background:#34495e;padding:30px;border-radius:12px;text-align:center;}</style></head><body>";
        echo "<div class='box'>";
        echo "<h2>Process Started</h2>";
        echo "<p>Process ID: <strong>{$this->processId}</strong></p>";
        echo "<p>Total Links: <strong>" . count($links) . "</strong></p>";
        echo "<p><a href='?process_id={$this->processId}' style='color:#3498db;'>Click here to see progress</a></p>";
        echo "</div></body></html>";
        exit;
    }

    /**
     * Process one batch of URLs
     */
    private function processBatch(): void
    {
        $result = $this->processor->processBatch();

        // Render progress
        $this->showProgress($result);

        // Auto-refresh if not complete
        if (!$result['complete']) {
            echo '<meta http-equiv="refresh" content="0">';
        } else {
            $this->tracker->cleanup();
            echo "<div style='text-align:center;margin-top:20px;'>";
            echo "<h2 style='color:#2ecc71;'>Process Complete!</h2>";
            echo "<p>Process <strong>{$this->processId}</strong> finished successfully.</p>";
            echo "<a href='?' style='color:#3498db;'>Start New Process</a>";
            echo "</div>";
        }
    }

    /**
     * Show the input form
     * v2.2: Added CSRF token generation
     */
    private function showForm(): void
    {
        $viewPath = $this->config['paths']['views'] . '/form.php';

        if (!file_exists($viewPath)) {
            throw new \Exception("View file not found: {$viewPath}");
        }

        $processId = $this->processId;
        $phases = $this->config['phases'];
        // SEC-006: Generate CSRF token for form
        $csrfToken = $this->generateCsrfToken();

        include $viewPath;
        exit;
    }

    /**
     * Show progress page (Dashboard)
     */
    private function showProgress(array $result): void
    {
        $viewPath = $this->config['paths']['views'] . '/progress.php';

        if (!file_exists($viewPath)) {
            throw new \Exception("View file not found: {$viewPath}");
        }

        $data = $result['data'];
        $logs = $result['logs'];
        $processId = $this->processId;
        $currentPage = 'dashboard';

        include $viewPath;
    }

    /**
     * Clean old temporary folders
     */
    private function cleanOldFolders(): void
    {
        $tempPath = $this->config['paths']['temp'];
        $lifetime = $this->config['processing']['temp_folder_lifetime'];

        foreach (glob($tempPath . '/*') as $folder) {
            if (is_dir($folder) && (time() - filemtime($folder)) > $lifetime) {
                array_map('unlink', glob("$folder/*.*"));
                @rmdir($folder);
            }
        }
    }

    /**
     * Get database instance
     */
    public function getDatabase(): Database
    {
        return $this->database;
    }

    /**
     * Get configuration
     */
    public function getConfig(): array
    {
        return $this->config;
    }
}
