<?php
/**
 * Domain Statistics Page - Flowb0t DCI v2.0 Professional
 * Variables available: $data, $domainStats, $processId, $currentPage
 * All text in ENGLISH
 */

$baseUrl = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');

// Calculate totals
$totalDomains = count($domainStats);
$totalUrls = $data['total_links'] ?? 0;
$totalSuccess = 0;
$totalErrors = 0;
$totalIgnored = 0;
$bestDomain = null;
$worstDomain = null;
$bestRate = 0;
$worstRate = 100;

foreach ($domainStats as $domain => $stats) {
    $totalSuccess += $stats['success'] ?? 0;
    $totalErrors += $stats['error'] ?? 0;
    $totalIgnored += $stats['ignored'] ?? 0;

    $domainTotal = $stats['total'] ?? 1;
    $successRate = round((($stats['success'] ?? 0) / $domainTotal) * 100, 1);

    if ($successRate > $bestRate && $domainTotal >= 3) {
        $bestRate = $successRate;
        $bestDomain = $domain;
    }
    if ($successRate < $worstRate && $domainTotal >= 3) {
        $worstRate = $successRate;
        $worstDomain = $domain;
    }
}

// Top 10 domains by total
$topDomains = array_slice($domainStats, 0, 10, true);
$maxDomainTotal = max(array_column($domainStats, 'total') ?: [1]);

// v2.1: Slowest domains
$slowestDomains = $domainStats;
uasort($slowestDomains, fn($a, $b) => ($b['avg_response_time'] ?? 0) <=> ($a['avg_response_time'] ?? 0));
$slowestDomains = array_slice($slowestDomains, 0, 5, true);

// v2.1: HTTP codes helper
$getCodeClass = function($code) {
    return match(true) {
        (int)$code >= 200 && (int)$code < 300 => 'success',
        (int)$code >= 300 && (int)$code < 400 => 'info',
        (int)$code >= 400 && (int)$code < 500 => 'warning',
        (int)$code >= 500 => 'error',
        default => 'info',
    };
};
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Domains - <?= htmlspecialchars($processId) ?></title>
    <link rel="stylesheet" href="<?= $baseUrl ?>/assets/css/style.css">
    <style>
        body {
            background: linear-gradient(135deg, #0F172A 0%, #1E293B 50%, #0F172A 100%);
            min-height: 100vh;
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/partials/navigation.php'; ?>

    <div class="dashboard compact">
        <!-- ============ HEADER ============ -->
        <div class="page-header glass-card">
            <h1>Domain Statistics</h1>
            <p>Detailed breakdown of processing results by domain</p>
        </div>

        <!-- ============ SUMMARY CARDS ============ -->
        <div class="dashboard-row four-cols">
            <div class="summary-card glass-card">
                <div class="summary-icon">&#127760;</div>
                <div class="summary-value"><?= number_format($totalDomains) ?></div>
                <div class="summary-label">Total Domains</div>
            </div>
            <div class="summary-card glass-card success">
                <div class="summary-icon">&#10004;</div>
                <div class="summary-value"><?= number_format($totalSuccess) ?></div>
                <div class="summary-label">Total Success</div>
            </div>
            <div class="summary-card glass-card warning">
                <div class="summary-icon">&#9888;</div>
                <div class="summary-value"><?= number_format($totalIgnored) ?></div>
                <div class="summary-label">Total Ignored</div>
            </div>
            <div class="summary-card glass-card error">
                <div class="summary-icon">&#10060;</div>
                <div class="summary-value"><?= number_format($totalErrors) ?></div>
                <div class="summary-label">Total Errors</div>
            </div>
        </div>

        <!-- ============ BEST/WORST PERFORMERS ============ -->
        <div class="dashboard-row two-cols">
            <?php if ($bestDomain): ?>
            <div class="performer-card glass-card success">
                <div class="performer-header">
                    <span class="performer-icon">&#127942;</span>
                    <span class="performer-title">Best Performer</span>
                </div>
                <div class="performer-domain"><?= htmlspecialchars($bestDomain) ?></div>
                <div class="performer-rate"><?= $bestRate ?>% success rate</div>
            </div>
            <?php endif; ?>
            <?php if ($worstDomain): ?>
            <div class="performer-card glass-card error">
                <div class="performer-header">
                    <span class="performer-icon">&#128680;</span>
                    <span class="performer-title">Most Problematic</span>
                </div>
                <div class="performer-domain"><?= htmlspecialchars($worstDomain) ?></div>
                <div class="performer-rate"><?= $worstRate ?>% success rate</div>
            </div>
            <?php endif; ?>
        </div>

        <!-- ============ v2.1: SLOWEST DOMAINS ============ -->
        <?php if (!empty($slowestDomains)): ?>
        <div class="slowest-domains glass-card">
            <h4>&#9201; Slowest Domains</h4>
            <div class="slowest-list">
                <?php foreach ($slowestDomains as $domain => $stats):
                    $avgTime = $stats['avg_response_time'] ?? 0;
                    if ($avgTime <= 0) continue;
                ?>
                <div class="slowest-row">
                    <span class="slowest-domain" title="<?= htmlspecialchars($domain) ?>"><?= htmlspecialchars(strlen($domain) > 30 ? substr($domain, 0, 30) . '...' : $domain) ?></span>
                    <span class="slowest-time <?= $avgTime > 5 ? 'critical' : ($avgTime > 3 ? 'warning' : '') ?>"><?= number_format($avgTime, 2) ?>s</span>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- ============ TOP 10 DOMAINS CHART ============ -->
        <div class="chart-container glass-card">
            <div class="chart-header">
                <h3>Top 10 Domains by Volume</h3>
            </div>
            <div class="bar-chart">
                <?php foreach ($topDomains as $domain => $stats):
                    $total = $stats['total'] ?? 0;
                    $success = $stats['success'] ?? 0;
                    $error = $stats['error'] ?? 0;
                    $ignored = $stats['ignored'] ?? 0;
                    $width = ($total / $maxDomainTotal) * 100;
                    $successWidth = ($success / max($total, 1)) * 100;
                    $errorWidth = ($error / max($total, 1)) * 100;
                    $ignoredWidth = ($ignored / max($total, 1)) * 100;
                ?>
                <div class="bar-row">
                    <div class="bar-label" title="<?= htmlspecialchars($domain) ?>"><?= htmlspecialchars(substr($domain, 0, 25)) ?><?= strlen($domain) > 25 ? '...' : '' ?></div>
                    <div class="bar-container">
                        <div class="bar-fill-stacked" style="width: <?= $width ?>%">
                            <div class="bar-segment success" style="width: <?= $successWidth ?>%" title="Success: <?= $success ?>"></div>
                            <div class="bar-segment warning" style="width: <?= $ignoredWidth ?>%" title="Ignored: <?= $ignored ?>"></div>
                            <div class="bar-segment error" style="width: <?= $errorWidth ?>%" title="Error: <?= $error ?>"></div>
                        </div>
                    </div>
                    <div class="bar-value"><?= number_format($total) ?></div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- ============ DOMAIN TABLE ============ -->
        <div class="table-container glass-card">
            <div class="table-header">
                <div class="table-title">
                    <h3>All Domains</h3>
                    <span class="table-count"><?= number_format($totalDomains) ?> domains</span>
                </div>
                <div class="table-actions">
                    <input type="text" id="domainSearch" class="search-input" placeholder="Search domains...">
                    <div class="export-buttons">
                        <a href="?process_id=<?= urlencode($processId) ?>&page=api&action=export&type=domains&format=csv" class="export-btn csv">CSV</a>
                        <a href="?process_id=<?= urlencode($processId) ?>&page=api&action=export&type=domains&format=json" class="export-btn json">JSON</a>
                    </div>
                </div>
            </div>
            <div class="table-responsive">
                <table class="data-table domain-table" id="domainTable">
                    <thead>
                        <tr>
                            <th class="sortable" data-sort="domain">Domain</th>
                            <th class="sortable" data-sort="total">Total</th>
                            <th class="sortable" data-sort="success">Success</th>
                            <th class="sortable" data-sort="error">Errors</th>
                            <th class="sortable" data-sort="ignored">Ignored</th>
                            <th class="sortable" data-sort="rate">Rate</th>
                            <th class="sortable" data-sort="avg_time">Avg Time</th>
                            <th>HTTP Codes</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($domainStats as $domain => $stats):
                            $total = $stats['total'] ?? 0;
                            $success = $stats['success'] ?? 0;
                            $error = $stats['error'] ?? 0;
                            $ignored = $stats['ignored'] ?? 0;
                            $avgTime = $stats['avg_response_time'] ?? 0;
                            $minTime = ($stats['min_response_time'] ?? PHP_FLOAT_MAX) < PHP_FLOAT_MAX ? $stats['min_response_time'] : 0;
                            $maxTime = $stats['max_response_time'] ?? 0;
                            $httpCodes = $stats['http_codes'] ?? [];
                            arsort($httpCodes);
                            $rate = round(($success / max($total, 1)) * 100, 1);

                            $rateClass = 'excellent';
                            if ($rate < 80) $rateClass = 'good';
                            if ($rate < 50) $rateClass = 'poor';
                            if ($rate < 25) $rateClass = 'critical';
                        ?>
                        <tr class="domain-row <?= $rate < 50 ? 'problematic' : ($rate > 80 ? 'excellent' : '') ?>"
                            data-domain="<?= htmlspecialchars($domain) ?>"
                            data-total="<?= $total ?>"
                            data-success="<?= $success ?>"
                            data-error="<?= $error ?>"
                            data-ignored="<?= $ignored ?>"
                            data-rate="<?= $rate ?>"
                            data-avg_time="<?= $avgTime ?>"
                            data-min_time="<?= $minTime ?>"
                            data-max_time="<?= $maxTime ?>">
                            <td class="domain-name">
                                <span class="domain-text"><?= htmlspecialchars($domain) ?></span>
                            </td>
                            <td class="text-center"><?= number_format($total) ?></td>
                            <td class="text-center text-success"><?= number_format($success) ?></td>
                            <td class="text-center text-error"><?= number_format($error) ?></td>
                            <td class="text-center text-warning"><?= number_format($ignored) ?></td>
                            <td class="text-center">
                                <span class="rate-badge <?= $rateClass ?>"><?= $rate ?>%</span>
                            </td>
                            <td class="text-center" title="Min: <?= number_format($minTime, 2) ?>s | Max: <?= number_format($maxTime, 2) ?>s">
                                <?= number_format($avgTime, 2) ?>s
                            </td>
                            <td class="http-codes-mini">
                                <?php foreach (array_slice($httpCodes, 0, 3, true) as $code => $count):
                                    $codeClass = $getCodeClass($code);
                                ?>
                                <span class="http-badge-mini <?= $codeClass ?>" title="HTTP <?= $code ?>: <?= $count ?> requests"><?= $code ?>:<?= $count ?></span>
                                <?php endforeach; ?>
                                <?php if (empty($httpCodes)): ?>
                                <span class="no-codes">-</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-center">
                                <a href="?process_id=<?= urlencode($processId) ?>&page=logs&domain=<?= urlencode($domain) ?>"
                                   class="action-link" title="View logs for this domain">&#128269;</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- ============ ERROR DISTRIBUTION ============ -->
        <?php
        // Aggregate error types across all domains
        $errorDistribution = [];
        foreach ($domainStats as $stats) {
            foreach ($stats['error_types'] ?? [] as $type => $count) {
                $errorDistribution[$type] = ($errorDistribution[$type] ?? 0) + $count;
            }
        }
        arsort($errorDistribution);
        $totalErrorCount = array_sum($errorDistribution) ?: 1;
        ?>
        <?php if (!empty($errorDistribution)): ?>
        <div class="chart-container glass-card">
            <div class="chart-header">
                <h3>Error Type Distribution</h3>
            </div>
            <div class="error-distribution">
                <?php foreach ($errorDistribution as $type => $count):
                    $percent = round(($count / $totalErrorCount) * 100, 1);
                    $label = match($type) {
                        'timeout' => 'Timeout',
                        'http_429' => 'HTTP 429 (Rate Limit)',
                        'http_404' => 'HTTP 404 (Not Found)',
                        'http_403' => 'HTTP 403 (Forbidden)',
                        'http_5xx' => 'Server Error (5xx)',
                        'connection' => 'Connection Error',
                        'metadata' => 'Metadata Error',
                        default => ucfirst($type),
                    };
                ?>
                <div class="error-row">
                    <div class="error-label"><?= $label ?></div>
                    <div class="error-bar-container">
                        <div class="error-bar" style="width: <?= $percent ?>%"></div>
                    </div>
                    <div class="error-count"><?= number_format($count) ?> (<?= $percent ?>%)</div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="<?= $baseUrl ?>/assets/js/app.js"></script>
    <script>
        window.processId = '<?= htmlspecialchars($processId) ?>';

        // Initialize domain table functionality
        document.addEventListener('DOMContentLoaded', function() {
            FlowbotDCI.initDomainTable();
        });
    </script>
</body>
</html>
