<?php
/**
 * FLOWB0T NEXUS - CLI Worker
 * Background worker process for crawling jobs
 * Run via cron: * * * * * /usr/bin/php /path/to/worker.php
 */

// Define root path
define('NEXUS_ROOT', dirname(__DIR__));
define('NEXUS_VERSION', '1.0.0');

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', NEXUS_ROOT . '/logs/error/worker.log');

// Set timezone
date_default_timezone_set('America/Sao_Paulo');

// Load environment
$envFile = NEXUS_ROOT . '/.env';
if (file_exists($envFile)) {
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '#') === 0) continue;
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
            putenv(trim($key) . '=' . trim($value));
        }
    }
}

// Load classes
require_once NEXUS_ROOT . '/core/Database.php';
require_once NEXUS_ROOT . '/core/Logger.php';
require_once NEXUS_ROOT . '/engine/CrawlerEngine.php';

use Flowb0t\Core\Database;
use Flowb0t\Core\Logger;
use Flowb0t\Engine\CrawlerEngine;

// Check for lock file
$lockFile = NEXUS_ROOT . '/temp/worker.lock';
if (file_exists($lockFile)) {
    $lockTime = filemtime($lockFile);
    // If lock is older than 30 minutes, remove it (stale lock)
    if (time() - $lockTime > 1800) {
        unlink($lockFile);
    } else {
        echo "Worker already running.\n";
        exit(0);
    }
}

// Create lock file
file_put_contents($lockFile, getmypid());

// Register shutdown function to remove lock
register_shutdown_function(function() use ($lockFile) {
    if (file_exists($lockFile)) {
        unlink($lockFile);
    }
});

// Initialize
$logger = new Logger('worker');
$logger->info('WORKER', 'Worker started');

try {
    $db = Database::getInstance();
    $pdo = $db->getConnection();

    // Get queued jobs
    $stmt = $pdo->query("
        SELECT job_uuid, name
        FROM nexus_jobs
        WHERE status = 'queued'
        ORDER BY priority DESC, created_at ASC
        LIMIT 1
    ");

    $job = $stmt->fetch();

    if (!$job) {
        $logger->debug('WORKER', 'No queued jobs found');
        exit(0);
    }

    $logger->info('WORKER', "Starting job: {$job['name']}", ['job_uuid' => $job['job_uuid']]);

    // Update job status to running
    $pdo->prepare("
        UPDATE nexus_jobs SET status = 'running', started_at = NOW()
        WHERE job_uuid = ?
    ")->execute([$job['job_uuid']]);

    // Run the crawler engine
    $engine = new CrawlerEngine($job['job_uuid']);
    $engine->start();

    $logger->info('WORKER', "Job completed: {$job['name']}");

} catch (Exception $e) {
    $logger->critical('WORKER', "Worker error: {$e->getMessage()}");

    // Mark job as failed if there was a job running
    if (isset($job) && $job) {
        try {
            $pdo->prepare("
                UPDATE nexus_jobs SET status = 'failed', completed_at = NOW()
                WHERE job_uuid = ?
            ")->execute([$job['job_uuid']]);
        } catch (Exception $e2) {}
    }
}

$logger->info('WORKER', 'Worker finished');
