<?php
/**
 * FLOWB0T NEXUS - Settings Page
 * System configuration and settings
 */

require_once NEXUS_ROOT . '/core/Database.php';
use Flowb0t\Core\Database;

$db = Database::getInstance();
$pdo = $db->getConnection();

// Get current settings
$settings = [];
try {
    $stmt = $pdo->query("SELECT * FROM nexus_settings ORDER BY setting_key");
    foreach ($stmt->fetchAll() as $row) {
        $settings[$row['setting_key']] = [
            'value' => $row['setting_value'],
            'type' => $row['setting_type'],
            'description' => $row['description']
        ];
    }
} catch (Exception $e) {}

// Default settings if table is empty
$defaultSettings = [
    'max_concurrent_jobs' => ['value' => '10', 'type' => 'int', 'description' => 'Maximum concurrent running jobs'],
    'max_concurrent_workers' => ['value' => '500', 'type' => 'int', 'description' => 'Maximum concurrent crawler workers'],
    'default_request_timeout' => ['value' => '10', 'type' => 'int', 'description' => 'Default HTTP request timeout in seconds'],
    'default_delay_between' => ['value' => '0.5', 'type' => 'float', 'description' => 'Default delay between requests in seconds'],
    'max_queue_size' => ['value' => '1000000', 'type' => 'int', 'description' => 'Maximum URLs in queue per job'],
    'auto_cleanup_days' => ['value' => '30', 'type' => 'int', 'description' => 'Auto-delete completed jobs after X days'],
    'export_retention_days' => ['value' => '7', 'type' => 'int', 'description' => 'Auto-delete export files after X days'],
    'log_retention_days' => ['value' => '14', 'type' => 'int', 'description' => 'Auto-delete logs after X days'],
];

$settings = array_merge($defaultSettings, $settings);

// Get system info
$phpVersion = phpversion();
$mysqlVersion = '';
try {
    $stmt = $pdo->query("SELECT VERSION()");
    $mysqlVersion = $stmt->fetchColumn();
} catch (Exception $e) {}

$memoryLimit = ini_get('memory_limit');
$maxExecution = ini_get('max_execution_time');
$curlVersion = curl_version()['version'] ?? 'N/A';
?>

<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="mb-0">Settings</h2>
        <small class="text-muted">Configure FLOWB0T NEXUS system settings</small>
    </div>
</div>

<div class="row g-4">
    <!-- Main Settings -->
    <div class="col-lg-8">
        <!-- Crawler Settings -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-robot me-2"></i>Crawler Settings</h5>
            </div>
            <div class="card-body">
                <form id="crawlerSettings">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">
                                Max Concurrent Jobs
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['max_concurrent_jobs']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="max_concurrent_jobs"
                                   value="<?= htmlspecialchars($settings['max_concurrent_jobs']['value']) ?>" min="1" max="100">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">
                                Max Concurrent Workers
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['max_concurrent_workers']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="max_concurrent_workers"
                                   value="<?= htmlspecialchars($settings['max_concurrent_workers']['value']) ?>" min="1" max="1000">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">
                                Default Request Timeout (seconds)
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['default_request_timeout']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="default_request_timeout"
                                   value="<?= htmlspecialchars($settings['default_request_timeout']['value']) ?>" min="1" max="60">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">
                                Default Delay Between (seconds)
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['default_delay_between']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="default_delay_between"
                                   value="<?= htmlspecialchars($settings['default_delay_between']['value']) ?>" min="0" max="10" step="0.1">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">
                                Max Queue Size per Job
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['max_queue_size']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="max_queue_size"
                                   value="<?= htmlspecialchars($settings['max_queue_size']['value']) ?>" min="1000">
                        </div>
                    </div>
                    <div class="mt-3">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save me-2"></i>Save Crawler Settings
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Cleanup Settings -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-trash me-2"></i>Cleanup Settings</h5>
            </div>
            <div class="card-body">
                <form id="cleanupSettings">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">
                                Auto-cleanup Jobs (days)
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['auto_cleanup_days']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="auto_cleanup_days"
                                   value="<?= htmlspecialchars($settings['auto_cleanup_days']['value']) ?>" min="1" max="365">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">
                                Export Retention (days)
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['export_retention_days']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="export_retention_days"
                                   value="<?= htmlspecialchars($settings['export_retention_days']['value']) ?>" min="1" max="90">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">
                                Log Retention (days)
                                <i class="bi bi-question-circle tooltip-icon" data-bs-toggle="tooltip"
                                   title="<?= htmlspecialchars($settings['log_retention_days']['description']) ?>"></i>
                            </label>
                            <input type="number" class="form-control" name="log_retention_days"
                                   value="<?= htmlspecialchars($settings['log_retention_days']['value']) ?>" min="1" max="90">
                        </div>
                    </div>
                    <div class="mt-3">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save me-2"></i>Save Cleanup Settings
                        </button>
                        <button type="button" class="btn btn-outline-warning ms-2" onclick="runCleanup()">
                            <i class="bi bi-trash me-2"></i>Run Cleanup Now
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Database Settings -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-database me-2"></i>Database Info</h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Database Name</label>
                        <input type="text" class="form-control" value="<?= getenv('DB_NAME') ?: 'digupdog_FEED' ?>" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">MySQL Version</label>
                        <input type="text" class="form-control" value="<?= htmlspecialchars($mysqlVersion) ?>" readonly>
                    </div>
                </div>
                <div class="mt-3">
                    <button type="button" class="btn btn-outline-info" onclick="optimizeTables()">
                        <i class="bi bi-speedometer me-2"></i>Optimize Tables
                    </button>
                    <button type="button" class="btn btn-outline-secondary ms-2" onclick="showTableStats()">
                        <i class="bi bi-bar-chart me-2"></i>Table Statistics
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Sidebar -->
    <div class="col-lg-4">
        <!-- System Info -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-cpu me-2"></i>System Info</h5>
            </div>
            <div class="card-body">
                <table class="table table-sm mb-0">
                    <tr>
                        <td class="text-muted">NEXUS Version</td>
                        <td><?= NEXUS_VERSION ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">PHP Version</td>
                        <td><?= $phpVersion ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">MySQL Version</td>
                        <td><?= $mysqlVersion ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">cURL Version</td>
                        <td><?= $curlVersion ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">Memory Limit</td>
                        <td><?= $memoryLimit ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">Max Execution</td>
                        <td><?= $maxExecution ?>s</td>
                    </tr>
                    <tr>
                        <td class="text-muted">Server Time</td>
                        <td><?= date('Y-m-d H:i:s') ?></td>
                    </tr>
                    <tr>
                        <td class="text-muted">Timezone</td>
                        <td><?= date_default_timezone_get() ?></td>
                    </tr>
                </table>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-lightning me-2"></i>Quick Actions</h5>
            </div>
            <div class="card-body d-grid gap-2">
                <button class="btn btn-outline-primary" onclick="testConnection()">
                    <i class="bi bi-plug me-2"></i>Test DB Connection
                </button>
                <button class="btn btn-outline-success" onclick="clearCache()">
                    <i class="bi bi-arrow-repeat me-2"></i>Clear Cache
                </button>
                <button class="btn btn-outline-warning" onclick="resetSettings()">
                    <i class="bi bi-arrow-counterclockwise me-2"></i>Reset to Defaults
                </button>
            </div>
        </div>

        <!-- Environment -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><i class="bi bi-file-earmark-code me-2"></i>Environment</h5>
                <button class="btn btn-sm btn-outline-light" onclick="toggleEnv()">
                    <i class="bi bi-eye"></i>
                </button>
            </div>
            <div class="card-body">
                <div id="envVars" style="display: none;">
                    <small class="text-muted">
                        <pre class="mb-0" style="font-size: 0.75rem;">
DB_HOST=<?= getenv('DB_HOST') ?: 'localhost' ?>

DB_NAME=<?= getenv('DB_NAME') ?: 'digupdog_FEED' ?>

DB_USER=<?= getenv('DB_USER') ?: '****' ?>

DB_PASS=****
NEXUS_ROOT=<?= NEXUS_ROOT ?>
                        </pre>
                    </small>
                </div>
                <p class="text-muted small mb-0" id="envHint">
                    <i class="bi bi-info-circle me-1"></i>Click the eye icon to view environment variables
                </p>
            </div>
        </div>
    </div>
</div>

<script>
// Initialize tooltips
document.querySelectorAll('[data-bs-toggle="tooltip"]').forEach(el => {
    new bootstrap.Tooltip(el);
});

// Save crawler settings
document.getElementById('crawlerSettings').addEventListener('submit', function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    const data = Object.fromEntries(formData);

    NEXUS.api('save_settings', {group: 'crawler', settings: data}).then(r => {
        if (r.success) {
            NEXUS.toast('Crawler settings saved', 'success');
        } else {
            NEXUS.toast('Failed to save settings', 'error');
        }
    });
});

// Save cleanup settings
document.getElementById('cleanupSettings').addEventListener('submit', function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    const data = Object.fromEntries(formData);

    NEXUS.api('save_settings', {group: 'cleanup', settings: data}).then(r => {
        if (r.success) {
            NEXUS.toast('Cleanup settings saved', 'success');
        } else {
            NEXUS.toast('Failed to save settings', 'error');
        }
    });
});

function runCleanup() {
    if (confirm('Run cleanup now? This will delete old jobs, exports, and logs based on retention settings.')) {
        NEXUS.api('run_cleanup', {}).then(r => {
            if (r.success) {
                NEXUS.toast('Cleanup completed: ' + r.message, 'success');
            } else {
                NEXUS.toast('Cleanup failed', 'error');
            }
        });
    }
}

function testConnection() {
    NEXUS.api('test_connection', {}).then(r => {
        if (r.success) {
            NEXUS.toast('Database connection OK', 'success');
        } else {
            NEXUS.toast('Connection failed: ' + r.error, 'error');
        }
    });
}

function clearCache() {
    NEXUS.api('clear_cache', {}).then(r => {
        NEXUS.toast('Cache cleared', 'success');
    });
}

function resetSettings() {
    if (confirm('Reset all settings to defaults? This cannot be undone.')) {
        NEXUS.api('reset_settings', {}).then(r => {
            if (r.success) {
                NEXUS.toast('Settings reset to defaults', 'success');
                location.reload();
            }
        });
    }
}

function optimizeTables() {
    NEXUS.toast('Optimizing tables...', 'info');
    NEXUS.api('optimize_tables', {}).then(r => {
        if (r.success) {
            NEXUS.toast('Tables optimized', 'success');
        } else {
            NEXUS.toast('Optimization failed', 'error');
        }
    });
}

function showTableStats() {
    NEXUS.api('table_stats', {}).then(r => {
        if (r.success) {
            alert(JSON.stringify(r.stats, null, 2));
        }
    });
}

function toggleEnv() {
    const env = document.getElementById('envVars');
    const hint = document.getElementById('envHint');
    if (env.style.display === 'none') {
        env.style.display = 'block';
        hint.style.display = 'none';
    } else {
        env.style.display = 'none';
        hint.style.display = 'block';
    }
}
</script>
