<?php
/**
 * ===========================================
 * FLOWBOT DCI - DEEP CRAWLER v7.0
 * ===========================================
 * Unified crawler with Bing search + Deep crawl
 * Beautiful modern UI + Functional code
 *
 * Based on proven working code
 * Rewritten for efficiency and aesthetics
 */

// ============================================
// SECTION 1: CONFIGURATION & DATABASE
// ============================================

error_reporting(E_ALL);
ini_set('display_errors', 0);
set_time_limit(0);
ini_set('memory_limit', '512M');

$currentPage = 'crawler';
$baseUrl = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');

// Load config
$configPath = dirname(__DIR__) . '/config/config.php';
$config = file_exists($configPath) ? require $configPath : [];

// Database connection
$pdo = null;
try {
    $dbConfig = $config['database'] ?? [];
    $dsn = sprintf(
        'mysql:host=%s;dbname=%s;charset=%s',
        $dbConfig['host'] ?? 'localhost',
        $dbConfig['name'] ?? 'digupdog_FEED',
        $dbConfig['charset'] ?? 'utf8mb4'
    );
    $pdo = new PDO($dsn, $dbConfig['user'] ?? '', $dbConfig['password'] ?? '', [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    // Continue without database - will show warning in UI
}

// User agents for rotation
$userAgents = [
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:122.0) Gecko/20100101 Firefox/122.0',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.2 Safari/605.1.15',
    'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.0.0 Safari/537.36',
];

// Excluded domains
$excludedDomains = [
    'facebook.com', 'twitter.com', 'instagram.com', 'linkedin.com',
    'pinterest.com', 'tiktok.com', 'reddit.com', 'tumblr.com',
    'bing.com', 'google.com', 'yahoo.com', 'duckduckgo.com',
    'youtube.com', 'youtu.be', 'vimeo.com',
];

// 4-Phase batch processing config
$phases = [
    ['concurrency' => 100, 'timeout' => 1],
    ['concurrency' => 60,  'timeout' => 2],
    ['concurrency' => 30,  'timeout' => 5],
    ['concurrency' => 10,  'timeout' => 10],
];

// Relevance scoring weights
$relevanceWeights = [
    'title' => 10,
    'h1' => 8,
    'h2' => 5,
    'description' => 6,
    'paragraph' => 3,
];

// ============================================
// SECTION 2: HELPER FUNCTIONS
// ============================================

/**
 * Build Yahoo search URL (Primary - works better from datacenter IPs)
 */
function buildYahooSearchUrl(string $query, int $offset = 0, string $type = 'web'): string {
    $baseUrls = [
        'web' => 'https://search.yahoo.com/search',
        'images' => 'https://images.search.yahoo.com/search/images',
        'videos' => 'https://video.search.yahoo.com/search/video',
    ];
    $params = ['p' => $query];
    if ($offset > 0) {
        $params['b'] = $offset + 1;
    }
    return ($baseUrls[$type] ?? $baseUrls['web']) . '?' . http_build_query($params);
}

/**
 * Build Bing search URL (Fallback)
 */
function buildBingSearchUrl(string $query, int $offset = 0, string $type = 'web'): string {
    $baseUrls = [
        'web' => 'https://www.bing.com/search',
        'images' => 'https://www.bing.com/images/search',
        'videos' => 'https://www.bing.com/videos/search',
    ];
    $params = ['q' => $query];
    if ($offset > 0) {
        $params['first'] = $offset + 1;
    }
    return ($baseUrls[$type] ?? $baseUrls['web']) . '?' . http_build_query($params);
}

/**
 * Fetch batch of URLs with cURL multi
 */
function fetchBatchWebInfo(array $urls, array $phases, array $userAgents): array {
    $results = [];
    $pending = array_map(fn($url) => ['url' => $url, 'phase' => 0, 'attempts' => 0], $urls);

    while (!empty($pending)) {
        $batch = [];
        $currentPhase = min(array_column($pending, 'phase'));
        $phaseConfig = $phases[$currentPhase] ?? end($phases);

        foreach ($pending as $key => $item) {
            if ($item['phase'] === $currentPhase && count($batch) < $phaseConfig['concurrency']) {
                $batch[$key] = $item;
            }
        }

        if (empty($batch)) break;

        $mh = curl_multi_init();
        $handles = [];

        foreach ($batch as $key => $item) {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $item['url'],
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => $phaseConfig['timeout'],
                CURLOPT_CONNECTTIMEOUT => min(3, $phaseConfig['timeout']),
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS => 5,
                CURLOPT_USERAGENT => $userAgents[array_rand($userAgents)],
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_HTTPHEADER => [
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: en-US,en;q=0.5',
                    'Accept-Encoding: gzip, deflate',
                    'Connection: keep-alive',
                ],
                CURLOPT_ENCODING => 'gzip, deflate',
            ]);
            curl_multi_add_handle($mh, $ch);
            $handles[$key] = $ch;
        }

        do {
            $status = curl_multi_exec($mh, $running);
            if ($running) {
                curl_multi_select($mh, 0.1);
            }
        } while ($running && $status === CURLM_OK);

        foreach ($handles as $key => $ch) {
            $response = curl_multi_getcontent($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);

            if ($httpCode === 200 && !empty($response)) {
                $results[$pending[$key]['url']] = [
                    'html' => $response,
                    'code' => $httpCode,
                    'success' => true,
                ];
                unset($pending[$key]);
            } else {
                $pending[$key]['phase']++;
                $pending[$key]['attempts']++;
                if ($pending[$key]['phase'] >= count($phases)) {
                    $results[$pending[$key]['url']] = [
                        'html' => '',
                        'code' => $httpCode,
                        'error' => $error ?: "HTTP $httpCode",
                        'success' => false,
                    ];
                    unset($pending[$key]);
                }
            }

            curl_multi_remove_handle($mh, $ch);
            curl_close($ch);
        }

        curl_multi_close($mh);
    }

    return $results;
}

/**
 * Calculate relevance score
 */
function calculateRelevanceScore(string $html, string $searchTerm, array $weights): int {
    if (empty($searchTerm)) return 100;

    $score = 0;
    $term = strtolower($searchTerm);
    $html = strtolower($html);

    // Title
    if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $html, $m)) {
        if (stripos($m[1], $term) !== false) {
            $score += $weights['title'];
        }
    }

    // H1
    if (preg_match_all('/<h1[^>]*>(.*?)<\/h1>/is', $html, $matches)) {
        foreach ($matches[1] as $h1) {
            if (stripos($h1, $term) !== false) {
                $score += $weights['h1'];
                break;
            }
        }
    }

    // H2
    if (preg_match_all('/<h2[^>]*>(.*?)<\/h2>/is', $html, $matches)) {
        $h2Count = 0;
        foreach ($matches[1] as $h2) {
            if (stripos($h2, $term) !== false) {
                $h2Count++;
                if ($h2Count >= 3) break;
            }
        }
        $score += $h2Count * $weights['h2'];
    }

    // Meta description
    if (preg_match('/<meta[^>]*name=["\']description["\'][^>]*content=["\'](.*?)["\']/is', $html, $m)) {
        if (stripos($m[1], $term) !== false) {
            $score += $weights['description'];
        }
    }

    // Paragraphs
    if (preg_match_all('/<p[^>]*>(.*?)<\/p>/is', $html, $matches)) {
        $pCount = 0;
        foreach ($matches[1] as $p) {
            if (stripos($p, $term) !== false) {
                $pCount++;
                if ($pCount >= 5) break;
            }
        }
        $score += $pCount * $weights['paragraph'];
    }

    return min(100, $score);
}

/**
 * Check if page is relevant
 */
function isRelevant(string $html, string $searchTerm): bool {
    if (empty($searchTerm)) return true;
    return stripos($html, $searchTerm) !== false;
}

/**
 * Extract links from HTML
 */
function extractLinks(string $html, string $baseUrl): array {
    $links = [];
    $baseParts = parse_url($baseUrl);
    $baseHost = $baseParts['host'] ?? '';
    $baseScheme = $baseParts['scheme'] ?? 'https';

    if (preg_match_all('/<a[^>]+href=["\']([^"\']+)["\']/i', $html, $matches)) {
        foreach ($matches[1] as $href) {
            $href = trim($href);
            if (empty($href) || $href[0] === '#' || stripos($href, 'javascript:') === 0 || stripos($href, 'mailto:') === 0) {
                continue;
            }

            // Normalize URL
            if (strpos($href, '//') === 0) {
                $href = $baseScheme . ':' . $href;
            } elseif ($href[0] === '/') {
                $href = $baseScheme . '://' . $baseHost . $href;
            } elseif (strpos($href, 'http') !== 0) {
                $basePath = dirname($baseParts['path'] ?? '/');
                $href = $baseScheme . '://' . $baseHost . $basePath . '/' . $href;
            }

            // Clean URL
            $href = preg_replace('/[#\?].*$/', '', $href);
            $href = rtrim($href, '/');

            if (filter_var($href, FILTER_VALIDATE_URL)) {
                $links[] = $href;
            }
        }
    }

    return array_unique($links);
}

/**
 * Check if URL is from excluded domain
 */
function isExcludedDomain(string $url, array $excludedDomains): bool {
    $host = parse_url($url, PHP_URL_HOST);
    if (!$host) return true;

    foreach ($excludedDomains as $excluded) {
        if (stripos($host, $excluded) !== false) {
            return true;
        }
    }
    return false;
}

/**
 * Parse Yahoo search results (Primary parser)
 */
function parseYahooResults(string $html): array {
    $results = [];
    $seen = [];

    // Pattern 1: Links through Yahoo redirect (most common)
    if (preg_match_all('/<a[^>]*href="(https:\/\/r\.search\.yahoo\.com[^"]+)"[^>]*>(?:<b>|<span[^>]*>)?([^<]+)/is', $html, $matches, PREG_SET_ORDER)) {
        foreach ($matches as $match) {
            // Decode Yahoo redirect URL
            if (preg_match('/RU=([^\/]+)/', $match[1], $ru)) {
                $realUrl = urldecode($ru[1]);
                $title = html_entity_decode(strip_tags($match[2]), ENT_QUOTES, 'UTF-8');

                if (!empty($realUrl) && !empty($title) && !isset($seen[$realUrl])) {
                    $seen[$realUrl] = true;
                    $results[] = [
                        'url' => $realUrl,
                        'title' => $title,
                    ];
                }
            }
        }
    }

    // Pattern 2: H3 titles with spans (for titles without direct links)
    if (preg_match_all('/<h3[^>]*class="[^"]*title[^"]*"[^>]*><span[^>]*>([^<]+)<\/span><\/h3>/is', $html, $matches)) {
        foreach ($matches[1] as $title) {
            $title = html_entity_decode(strip_tags($title), ENT_QUOTES, 'UTF-8');
            // Store title for matching later if needed
        }
    }

    return $results;
}

/**
 * Parse Bing search results (Fallback parser)
 */
function parseBingResults(string $html): array {
    $results = [];

    // Check for CAPTCHA - if present, Bing is blocking us
    if (stripos($html, 'captcha') !== false || stripos($html, 'turnstile') !== false) {
        return []; // Return empty to trigger fallback
    }

    // Parse web results
    if (preg_match_all('/<li class="b_algo"[^>]*>(.*?)<\/li>/is', $html, $matches)) {
        foreach ($matches[1] as $item) {
            $url = '';
            $title = '';

            if (preg_match('/<a[^>]+href=["\'](https?:\/\/[^"\']+)["\'][^>]*>(.*?)<\/a>/is', $item, $m)) {
                $url = $m[1];
                $title = strip_tags($m[2]);
            }

            if (!empty($url) && !empty($title)) {
                $results[] = [
                    'url' => $url,
                    'title' => html_entity_decode($title, ENT_QUOTES, 'UTF-8'),
                ];
            }
        }
    }

    return $results;
}

// ============================================
// SECTION 3: PROCESS FORM SUBMISSION
// ============================================

$isProcessing = false;
$progress = null;
$results = [];
$errors = [];
$logs = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];

    if ($action === 'search') {
        $isProcessing = true;
        $query = trim($_POST['query'] ?? '');
        $maxResults = min(500, max(10, intval($_POST['max_results'] ?? 100)));
        $searchType = $_POST['search_type'] ?? 'web';
        $deepCrawl = isset($_POST['deep_crawl']);
        $maxDepth = min(5, max(1, intval($_POST['max_depth'] ?? 2)));
        $sameDomain = isset($_POST['same_domain']);

        if (empty($query)) {
            $errors[] = 'Please enter a search query';
        } else {
            $logs[] = ['type' => 'info', 'message' => "Starting Yahoo search for: \"$query\""];

            // Search Yahoo (primary - works from datacenter IPs)
            $searchUrls = [];
            $offset = 0;
            $pages = ceil($maxResults / 7); // Yahoo shows ~7 results per page

            for ($i = 0; $i < $pages; $i++) {
                $searchUrls[] = buildYahooSearchUrl($query, $offset, $searchType);
                $offset += 7;
            }

            $logs[] = ['type' => 'info', 'message' => "Fetching " . count($searchUrls) . " search pages..."];

            $searchResults = fetchBatchWebInfo($searchUrls, $phases, $userAgents);

            $foundUrls = [];
            foreach ($searchResults as $url => $result) {
                if ($result['success'] && !empty($result['html'])) {
                    $parsed = parseYahooResults($result['html']);
                    foreach ($parsed as $item) {
                        if (!isExcludedDomain($item['url'], $excludedDomains)) {
                            $foundUrls[$item['url']] = $item;
                        }
                    }
                }
            }

            // Fallback to Bing if Yahoo returns few results
            if (count($foundUrls) < 5) {
                $logs[] = ['type' => 'warning', 'message' => "Yahoo returned few results, trying Bing..."];

                $searchUrls = [];
                $offset = 0;
                $pages = ceil($maxResults / 10);

                for ($i = 0; $i < $pages; $i++) {
                    $searchUrls[] = buildBingSearchUrl($query, $offset, $searchType);
                    $offset += 10;
                }

                $searchResults = fetchBatchWebInfo($searchUrls, $phases, $userAgents);

                foreach ($searchResults as $url => $result) {
                    if ($result['success'] && !empty($result['html'])) {
                        $parsed = parseBingResults($result['html']);
                        foreach ($parsed as $item) {
                            if (!isExcludedDomain($item['url'], $excludedDomains)) {
                                $foundUrls[$item['url']] = $item;
                            }
                        }
                    }
                }
            }

            $logs[] = ['type' => 'success', 'message' => "Found " . count($foundUrls) . " URLs from search"];

            // Fetch found URLs for relevance scoring
            if (!empty($foundUrls)) {
                $logs[] = ['type' => 'info', 'message' => "Fetching pages for relevance scoring..."];

                $urlsToFetch = array_keys(array_slice($foundUrls, 0, $maxResults));
                $pageResults = fetchBatchWebInfo($urlsToFetch, $phases, $userAgents);

                foreach ($pageResults as $url => $result) {
                    if ($result['success'] && !empty($result['html'])) {
                        $score = calculateRelevanceScore($result['html'], $query, $relevanceWeights);
                        $foundUrls[$url]['score'] = $score;
                        $foundUrls[$url]['relevant'] = isRelevant($result['html'], $query);

                        // Extract title if not already set
                        if (empty($foundUrls[$url]['title'])) {
                            if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $result['html'], $m)) {
                                $foundUrls[$url]['title'] = html_entity_decode(trim($m[1]), ENT_QUOTES, 'UTF-8');
                            }
                        }

                        // Deep crawl if enabled
                        if ($deepCrawl && $foundUrls[$url]['relevant']) {
                            $childLinks = extractLinks($result['html'], $url);
                            $foundUrls[$url]['children'] = count($childLinks);
                        }
                    } else {
                        $foundUrls[$url]['score'] = 0;
                        $foundUrls[$url]['relevant'] = false;
                        $foundUrls[$url]['error'] = $result['error'] ?? 'Failed to fetch';
                    }
                }
            }

            // Sort by score
            uasort($foundUrls, fn($a, $b) => ($b['score'] ?? 0) <=> ($a['score'] ?? 0));

            $results = array_values($foundUrls);
            $logs[] = ['type' => 'done', 'message' => "Search complete! Found " . count($results) . " results"];
        }
    }

    // URL Mode - Direct URL input
    if ($action === 'crawl_urls' && !empty($_POST['urls'])) {
        $isProcessing = true;
        $urlsText = trim($_POST['urls'] ?? '');
        $searchTerm = trim($_POST['url_search_term'] ?? '');

        // Parse URLs from textarea
        $inputUrls = array_filter(array_map('trim', explode("\n", $urlsText)));
        $validUrls = [];

        foreach ($inputUrls as $url) {
            $url = trim($url);
            if (filter_var($url, FILTER_VALIDATE_URL)) {
                $validUrls[] = $url;
            }
        }

        if (empty($validUrls)) {
            $errors[] = 'No valid URLs provided';
        } else {
            $logs[] = ['type' => 'info', 'message' => "Processing " . count($validUrls) . " URLs..."];

            // Fetch URLs
            $pageResults = fetchBatchWebInfo($validUrls, $phases, $userAgents);

            $foundUrls = [];
            foreach ($pageResults as $url => $result) {
                if ($result['success'] && !empty($result['html'])) {
                    $title = '';
                    if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $result['html'], $m)) {
                        $title = html_entity_decode(trim($m[1]), ENT_QUOTES, 'UTF-8');
                    }

                    $score = 0;
                    $relevant = true;
                    if (!empty($searchTerm)) {
                        $score = calculateRelevanceScore($result['html'], $searchTerm, $relevanceWeights);
                        $relevant = isRelevant($result['html'], $searchTerm);
                    }

                    $foundUrls[$url] = [
                        'url' => $url,
                        'title' => $title ?: $url,
                        'score' => $score,
                        'relevant' => $relevant,
                    ];

                    $logs[] = ['type' => 'success', 'message' => "Fetched: " . substr($title ?: $url, 0, 60)];
                } else {
                    $logs[] = ['type' => 'error', 'message' => "Failed: $url - " . ($result['error'] ?? 'Unknown error')];
                }
            }

            // Sort by score if search term provided
            if (!empty($searchTerm)) {
                uasort($foundUrls, fn($a, $b) => ($b['score'] ?? 0) <=> ($a['score'] ?? 0));
            }

            $results = array_values($foundUrls);
            $logs[] = ['type' => 'done', 'message' => "Crawl complete! Fetched " . count($results) . " pages"];
        }
    }

    if ($action === 'import' && !empty($_POST['urls'])) {
        $urlsToImport = $_POST['urls'];
        $imported = 0;
        $failed = 0;

        if ($pdo) {
            foreach ($urlsToImport as $url) {
                try {
                    // Check if already exists
                    $stmt = $pdo->prepare("SELECT id FROM pinfeeds WHERE url = ?");
                    $stmt->execute([$url]);
                    if ($stmt->fetch()) {
                        $failed++;
                        continue;
                    }

                    // Insert new
                    $stmt = $pdo->prepare("INSERT INTO pinfeeds (url, status, created_at) VALUES (?, 'pending', NOW())");
                    $stmt->execute([$url]);
                    $imported++;
                } catch (PDOException $e) {
                    $failed++;
                }
            }
            $logs[] = ['type' => 'success', 'message' => "Imported $imported URLs, $failed skipped/failed"];
        } else {
            $errors[] = 'Database not connected';
        }
    }
}

// ============================================
// SECTION 4: HTML OUTPUT
// ============================================
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Deep Crawler v7.0 - Flowb0t DCI</title>
    <style>
        :root {
            --bg-primary: #0f0f23;
            --bg-secondary: #1a1a2e;
            --bg-card: rgba(255, 255, 255, 0.05);
            --border-color: rgba(255, 255, 255, 0.1);
            --text-primary: #ffffff;
            --text-secondary: #a0a0b0;
            --text-muted: #6b6b7b;
            --accent-blue: #60a5fa;
            --accent-green: #34d399;
            --accent-red: #f87171;
            --accent-yellow: #fbbf24;
            --accent-purple: #a78bfa;
            --gradient-primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, var(--bg-primary) 0%, var(--bg-secondary) 100%);
            color: var(--text-primary);
            min-height: 100vh;
            line-height: 1.6;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 24px;
        }

        /* Header */
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 32px;
            padding-bottom: 24px;
            border-bottom: 1px solid var(--border-color);
        }

        .header h1 {
            font-size: 28px;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .header h1 .icon {
            width: 36px;
            height: 36px;
            background: var(--gradient-primary);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .badge {
            font-size: 12px;
            font-weight: 600;
            padding: 4px 12px;
            background: var(--gradient-primary);
            border-radius: 20px;
        }

        .back-link {
            color: var(--text-secondary);
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.2s;
        }

        .back-link:hover {
            color: var(--text-primary);
        }

        /* Cards */
        .card {
            background: var(--bg-card);
            backdrop-filter: blur(10px);
            border: 1px solid var(--border-color);
            border-radius: 16px;
            padding: 24px;
            margin-bottom: 24px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .card-title {
            font-size: 18px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        /* Form */
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            margin-bottom: 16px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }

        .form-group.full-width {
            grid-column: 1 / -1;
        }

        .form-group label {
            font-size: 13px;
            font-weight: 500;
            color: var(--text-secondary);
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group select {
            padding: 12px 16px;
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid var(--border-color);
            border-radius: 10px;
            color: var(--text-primary);
            font-size: 14px;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.2);
        }

        .form-group input::placeholder {
            color: var(--text-muted);
        }

        .checkbox-group {
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
        }

        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
        }

        .checkbox-item input[type="checkbox"] {
            width: 18px;
            height: 18px;
            accent-color: #667eea;
        }

        /* Buttons */
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.2s;
        }

        .btn-primary {
            background: var(--gradient-primary);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }

        .btn-secondary:hover {
            background: rgba(255, 255, 255, 0.15);
        }

        .btn-success {
            background: linear-gradient(135deg, #34d399, #10b981);
            color: white;
        }

        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(52, 211, 153, 0.4);
        }

        .form-actions {
            display: flex;
            gap: 12px;
            margin-top: 20px;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }

        .stat-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            border-left: 4px solid transparent;
        }

        .stat-card.blue { border-left-color: var(--accent-blue); }
        .stat-card.green { border-left-color: var(--accent-green); }
        .stat-card.red { border-left-color: var(--accent-red); }
        .stat-card.yellow { border-left-color: var(--accent-yellow); }

        .stat-value {
            font-size: 36px;
            font-weight: 700;
            margin-bottom: 4px;
        }

        .stat-card.blue .stat-value { color: var(--accent-blue); }
        .stat-card.green .stat-value { color: var(--accent-green); }
        .stat-card.red .stat-value { color: var(--accent-red); }
        .stat-card.yellow .stat-value { color: var(--accent-yellow); }

        .stat-label {
            font-size: 12px;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Activity Log */
        .activity-log {
            background: rgba(0, 0, 0, 0.3);
            border-radius: 10px;
            max-height: 250px;
            overflow-y: auto;
            padding: 12px;
        }

        .log-entry {
            display: flex;
            align-items: flex-start;
            gap: 12px;
            padding: 8px 12px;
            border-radius: 6px;
            margin-bottom: 4px;
            font-size: 13px;
            font-family: 'Monaco', 'Consolas', monospace;
        }

        .log-entry.info { background: rgba(96, 165, 250, 0.1); color: var(--accent-blue); }
        .log-entry.success { background: rgba(52, 211, 153, 0.1); color: var(--accent-green); }
        .log-entry.error { background: rgba(248, 113, 113, 0.1); color: var(--accent-red); }
        .log-entry.done { background: rgba(167, 139, 250, 0.1); color: var(--accent-purple); }

        .log-time {
            color: var(--text-muted);
            font-size: 11px;
            white-space: nowrap;
        }

        /* Results Table */
        .results-table {
            width: 100%;
            border-collapse: collapse;
        }

        .results-table th,
        .results-table td {
            padding: 12px 16px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        .results-table th {
            font-size: 12px;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            background: rgba(0, 0, 0, 0.2);
        }

        .results-table td {
            font-size: 13px;
        }

        .results-table tr:hover {
            background: rgba(255, 255, 255, 0.02);
        }

        .results-table .url-cell {
            max-width: 400px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }

        .results-table .url-cell a {
            color: var(--accent-blue);
            text-decoration: none;
        }

        .results-table .url-cell a:hover {
            text-decoration: underline;
        }

        .score-badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        .score-high { background: rgba(52, 211, 153, 0.2); color: var(--accent-green); }
        .score-medium { background: rgba(251, 191, 36, 0.2); color: var(--accent-yellow); }
        .score-low { background: rgba(248, 113, 113, 0.2); color: var(--accent-red); }

        .results-scroll {
            max-height: 500px;
            overflow-y: auto;
        }

        /* Empty state */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--text-secondary);
        }

        .empty-state .icon {
            font-size: 48px;
            margin-bottom: 16px;
            opacity: 0.5;
        }

        /* Alerts */
        .alert {
            padding: 16px 20px;
            border-radius: 10px;
            margin-bottom: 16px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .alert-error {
            background: rgba(248, 113, 113, 0.1);
            border: 1px solid rgba(248, 113, 113, 0.3);
            color: var(--accent-red);
        }

        .alert-warning {
            background: rgba(251, 191, 36, 0.1);
            border: 1px solid rgba(251, 191, 36, 0.3);
            color: var(--accent-yellow);
        }

        .alert-info {
            background: rgba(96, 165, 250, 0.1);
            border: 1px solid rgba(96, 165, 250, 0.3);
            color: var(--accent-blue);
            font-size: 13px;
        }

        .alert-success {
            background: rgba(52, 211, 153, 0.1);
            border: 1px solid rgba(52, 211, 153, 0.3);
            color: var(--accent-green);
        }

        /* Responsive */
        @media (max-width: 768px) {
            .container {
                padding: 16px;
            }

            .header h1 {
                font-size: 22px;
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <h1>
                <span class="icon">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"/>
                        <path d="M2 12h20M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/>
                    </svg>
                </span>
                Deep Crawler
                <span class="badge">v7.0</span>
            </h1>
            <a href="<?= $baseUrl ?>/" class="back-link">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M19 12H5M12 19l-7-7 7-7"/>
                </svg>
                Back to Dashboard
            </a>
        </div>

        <?php if (!$pdo): ?>
        <div class="alert alert-warning">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                <line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/>
            </svg>
            Database not connected. Import feature disabled.
        </div>
        <?php endif; ?>

        <?php foreach ($errors as $error): ?>
        <div class="alert alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/>
            </svg>
            <?= htmlspecialchars($error) ?>
        </div>
        <?php endforeach; ?>

        <!-- Search Form -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="11" cy="11" r="8"/><line x1="21" y1="21" x2="16.65" y2="16.65"/>
                    </svg>
                    Search Configuration
                </h2>
            </div>

            <form method="POST" action="">
                <input type="hidden" name="action" value="search">

                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="query">Search Query</label>
                        <input type="text" id="query" name="query" placeholder="Enter your search term (e.g., Batman the Movie)"
                               value="<?= htmlspecialchars($_POST['query'] ?? '') ?>" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="max_results">Max Results</label>
                        <input type="number" id="max_results" name="max_results" value="<?= $_POST['max_results'] ?? 100 ?>" min="10" max="500">
                    </div>
                    <div class="form-group">
                        <label for="search_type">Search Type</label>
                        <select id="search_type" name="search_type">
                            <option value="web" <?= ($_POST['search_type'] ?? 'web') === 'web' ? 'selected' : '' ?>>Web</option>
                            <option value="images" <?= ($_POST['search_type'] ?? '') === 'images' ? 'selected' : '' ?>>Images</option>
                            <option value="videos" <?= ($_POST['search_type'] ?? '') === 'videos' ? 'selected' : '' ?>>Videos</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="max_depth">Max Depth (Deep Crawl)</label>
                        <input type="number" id="max_depth" name="max_depth" value="<?= $_POST['max_depth'] ?? 2 ?>" min="1" max="5">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group full-width">
                        <label>Options</label>
                        <div class="checkbox-group">
                            <label class="checkbox-item">
                                <input type="checkbox" name="deep_crawl" <?= isset($_POST['deep_crawl']) ? 'checked' : '' ?>>
                                <span>Enable Deep Crawl</span>
                            </label>
                            <label class="checkbox-item">
                                <input type="checkbox" name="same_domain" <?= isset($_POST['same_domain']) ? 'checked' : '' ?>>
                                <span>Same Domain Only</span>
                            </label>
                        </div>
                    </div>
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="11" cy="11" r="8"/><line x1="21" y1="21" x2="16.65" y2="16.65"/>
                        </svg>
                        Search (Yahoo/Bing)
                    </button>
                </div>
                <div class="alert alert-info" style="margin-top: 16px;">
                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/>
                    </svg>
                    <span>Note: Search may be limited from datacenter IPs. Use URL Mode below if search fails.</span>
                </div>
            </form>
        </div>

        <!-- URL Mode -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/>
                        <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/>
                    </svg>
                    URL Mode (Direct Input)
                </h2>
            </div>
            <form method="POST" action="">
                <input type="hidden" name="action" value="crawl_urls">
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="urls">URLs to Crawl (one per line)</label>
                        <textarea id="urls" name="urls" rows="6"
                                  style="width: 100%; padding: 12px 16px; background: rgba(0, 0, 0, 0.3); border: 1px solid var(--border-color); border-radius: 10px; color: var(--text-primary); font-size: 14px; resize: vertical;"
                                  placeholder="https://example.com/page1&#10;https://example.com/page2&#10;https://another-site.com/article"><?= htmlspecialchars($_POST['urls'] ?? '') ?></textarea>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="url_search_term">Search Term (for relevance)</label>
                        <input type="text" id="url_search_term" name="url_search_term"
                               placeholder="Optional: term to score relevance"
                               value="<?= htmlspecialchars($_POST['url_search_term'] ?? '') ?>">
                    </div>
                </div>
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/>
                        </svg>
                        Crawl URLs
                    </button>
                </div>
            </form>
        </div>

        <?php if (!empty($logs)): ?>
        <!-- Activity Log -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                        <polyline points="14 2 14 8 20 8"/><line x1="16" y1="13" x2="8" y2="13"/><line x1="16" y1="17" x2="8" y2="17"/>
                    </svg>
                    Activity Log
                </h2>
            </div>
            <div class="activity-log">
                <?php foreach ($logs as $log): ?>
                <div class="log-entry <?= $log['type'] ?>">
                    <span class="log-time"><?= date('H:i:s') ?></span>
                    <span><?= htmlspecialchars($log['message']) ?></span>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($results)): ?>
        <!-- Stats -->
        <div class="stats-grid">
            <div class="stat-card blue">
                <div class="stat-value"><?= count($results) ?></div>
                <div class="stat-label">Total Results</div>
            </div>
            <div class="stat-card green">
                <div class="stat-value"><?= count(array_filter($results, fn($r) => ($r['score'] ?? 0) >= 50)) ?></div>
                <div class="stat-label">High Relevance</div>
            </div>
            <div class="stat-card yellow">
                <div class="stat-value"><?= count(array_filter($results, fn($r) => ($r['score'] ?? 0) >= 20 && ($r['score'] ?? 0) < 50)) ?></div>
                <div class="stat-label">Medium Relevance</div>
            </div>
            <div class="stat-card red">
                <div class="stat-value"><?= count(array_filter($results, fn($r) => ($r['score'] ?? 0) < 20)) ?></div>
                <div class="stat-label">Low Relevance</div>
            </div>
        </div>

        <!-- Results -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/>
                    </svg>
                    Search Results (<?= count($results) ?>)
                </h2>
                <?php if ($pdo): ?>
                <form method="POST" action="" id="importForm">
                    <input type="hidden" name="action" value="import">
                    <button type="submit" class="btn btn-success">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                            <polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/>
                        </svg>
                        Import Selected
                    </button>
                <?php endif; ?>
            </div>

            <div class="results-scroll">
                <table class="results-table">
                    <thead>
                        <tr>
                            <?php if ($pdo): ?>
                            <th style="width: 40px;">
                                <input type="checkbox" id="selectAll" onchange="toggleAll(this)">
                            </th>
                            <?php endif; ?>
                            <th>Title / URL</th>
                            <th style="width: 100px;">Score</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($results as $i => $result): ?>
                        <tr>
                            <?php if ($pdo): ?>
                            <td>
                                <input type="checkbox" name="urls[]" value="<?= htmlspecialchars($result['url']) ?>"
                                       <?= ($result['score'] ?? 0) >= 20 ? 'checked' : '' ?>>
                            </td>
                            <?php endif; ?>
                            <td class="url-cell">
                                <div><strong><?= htmlspecialchars($result['title'] ?? 'No title') ?></strong></div>
                                <a href="<?= htmlspecialchars($result['url']) ?>" target="_blank" rel="noopener">
                                    <?= htmlspecialchars($result['url']) ?>
                                </a>
                            </td>
                            <td>
                                <?php
                                $score = $result['score'] ?? 0;
                                $scoreClass = $score >= 50 ? 'score-high' : ($score >= 20 ? 'score-medium' : 'score-low');
                                ?>
                                <span class="score-badge <?= $scoreClass ?>"><?= $score ?></span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($pdo): ?>
                </form>
            <?php endif; ?>
        </div>
        <?php elseif ($isProcessing): ?>
        <!-- Empty state after processing with no results -->
        <div class="card">
            <div class="empty-state">
                <div class="icon">&#128269;</div>
                <h3>No results found</h3>
                <p>Try a different search query or adjust your settings</p>
            </div>
        </div>
        <?php else: ?>
        <!-- Initial empty state -->
        <div class="card">
            <div class="empty-state">
                <div class="icon">&#127758;</div>
                <h3>Ready to Search</h3>
                <p>Enter a search query above to find URLs using Bing search engine</p>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script>
    function toggleAll(checkbox) {
        const checkboxes = document.querySelectorAll('input[name="urls[]"]');
        checkboxes.forEach(cb => cb.checked = checkbox.checked);
    }

    // Auto-scroll activity log
    const log = document.querySelector('.activity-log');
    if (log) {
        log.scrollTop = log.scrollHeight;
    }
    </script>
</body>
</html>
