<?php
/**
 * ============================================
 * DCI CONTROL CENTER v1.0
 * ============================================
 * Unified management page for ALL crawler processes
 * - Global visibility of all modes
 * - Real-time updates via SSE
 * - Safe controls (pause/resume/stop/cleanup)
 * - Configuration management
 * - Logs and metrics dashboard
 * ============================================
 */

// ============================================
// CONFIGURATION
// ============================================
ini_set('display_errors', 1);
error_reporting(E_ALL);
set_time_limit(0);
ini_set('memory_limit', '512M');

$DB_HOST = 'localhost';
$DB_NAME = 'digupdog_FEED';
$DB_USER = 'digupdog_FEEDadmin';
$DB_PASS = 'Raimundinho1';

try {
    $pdo = new PDO(
        "mysql:host=$DB_HOST;dbname=$DB_NAME;charset=utf8mb4",
        $DB_USER,
        $DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Ensure crawler_settings table exists
try {
    $pdo->exec("CREATE TABLE IF NOT EXISTS crawler_settings (
        id INT PRIMARY KEY AUTO_INCREMENT,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )");
} catch (PDOException $e) {
    // Table might already exist
}

// Ensure crawler_alerts table exists
try {
    $pdo->exec("CREATE TABLE IF NOT EXISTS crawler_alerts (
        id INT PRIMARY KEY AUTO_INCREMENT,
        process_id VARCHAR(32),
        alert_type ENUM('error', 'warning', 'info') DEFAULT 'info',
        message TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        acknowledged BOOLEAN DEFAULT FALSE
    )");
} catch (PDOException $e) {
    // Table might already exist
}

// CSRF Token
session_start();
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrfToken = $_SESSION['csrf_token'];
session_write_close();

// Handle API Actions
$action = $_REQUEST['action'] ?? '';

// ============================================
// DCI SSE - Unified Data Stream
// ============================================
if ($action === 'dci_sse') {
    ignore_user_abort(true);
    @ini_set('zlib.output_compression', 0);
    @ini_set('output_buffering', 0);
    header('Content-Type: text/event-stream');
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');
    header('X-Accel-Buffering: no');
    header('Content-Encoding: none');
    while (ob_get_level()) ob_end_clean();
    ob_implicit_flush(true);

    $padding = "\n: " . str_repeat(' ', 2048) . "\n";

    while (!connection_aborted()) {
        $data = [];

        // ALL active processes (all modes)
        $procs = $pdo->query("SELECT id, mode, status, keywords, discovered, processed, imported, duplicates, errors, skipped, pool_size, waves, rate, level_stats, started_at, stopped_at, last_heartbeat, created_at FROM crawler_processes WHERE status IN ('running','paused','pending') ORDER BY created_at DESC LIMIT 200")->fetchAll(PDO::FETCH_ASSOC);

        // Recently completed/stopped (last 30 min)
        $recent = $pdo->query("SELECT id, mode, status, discovered, processed, imported, duplicates, errors, skipped, pool_size, waves, rate, started_at, stopped_at FROM crawler_processes WHERE status IN ('completed','stopped','dead') AND (stopped_at > DATE_SUB(NOW(), INTERVAL 30 MINUTE) OR stopped_at IS NULL) ORDER BY COALESCE(stopped_at, created_at) DESC LIMIT 50")->fetchAll(PDO::FETCH_ASSOC);

        // Aggregate stats
        $data['totals'] = [
            'total_processes' => count($procs),
            'running' => 0,
            'paused' => 0,
            'pending' => 0,
            'imported' => 0,
            'discovered' => 0,
            'processed' => 0,
            'duplicates' => 0,
            'errors' => 0,
            'skipped' => 0,
            'rate' => 0,
            'pool_size' => 0
        ];

        $data['modes'] = ['viral' => 0, 'ultimate' => 0, 'multi' => 0, 'infinite' => 0, 'search' => 0, 'deep' => 0, 'pagination' => 0, 'hybrid' => 0];

        foreach ($procs as &$p) {
            if ($p['status'] === 'running') $data['totals']['running']++;
            elseif ($p['status'] === 'paused') $data['totals']['paused']++;
            elseif ($p['status'] === 'pending') $data['totals']['pending']++;

            $data['totals']['imported'] += (int)$p['imported'];
            $data['totals']['discovered'] += (int)$p['discovered'];
            $data['totals']['processed'] += (int)$p['processed'];
            $data['totals']['duplicates'] += (int)$p['duplicates'];
            $data['totals']['errors'] += (int)$p['errors'];
            $data['totals']['skipped'] += (int)$p['skipped'];
            $data['totals']['rate'] += (float)$p['rate'];
            $data['totals']['pool_size'] += (int)$p['pool_size'];

            // Mode distribution
            $mode = strtolower($p['mode'] ?? 'viral');
            if (isset($data['modes'][$mode])) {
                $data['modes'][$mode]++;
            }

            // Calculate elapsed time
            $p['elapsed'] = $p['started_at'] ? max(1, time() - strtotime($p['started_at'])) : 0;

            // Parse keywords for display
            $keywords = json_decode($p['keywords'], true) ?: [];
            $p['keywords_preview'] = is_array($keywords) ? implode(', ', array_slice($keywords, 0, 3)) : substr($p['keywords'], 0, 50);
            if (count($keywords) > 3) $p['keywords_preview'] .= '...';

            // Check if dead (heartbeat > 45s)
            if ($p['status'] === 'running' && $p['last_heartbeat']) {
                $hbAge = time() - strtotime($p['last_heartbeat']);
                $p['heartbeat_age'] = $hbAge;
                if ($hbAge > 45) {
                    $p['suspected_dead'] = true;
                }
            }
        }
        unset($p);

        $data['processes'] = $procs;
        $data['recent'] = $recent;

        // Pool stats
        $data['pool'] = [
            'pending' => (int)$pdo->query("SELECT COUNT(*) FROM crawler_pool WHERE status='pending'")->fetchColumn(),
            'claimed' => (int)$pdo->query("SELECT COUNT(*) FROM crawler_pool WHERE status='claimed'")->fetchColumn(),
            'done' => (int)$pdo->query("SELECT COUNT(*) FROM crawler_pool WHERE status='done'")->fetchColumn()
        ];
        $data['seen_total'] = (int)$pdo->query("SELECT COUNT(*) FROM crawler_seen")->fetchColumn();

        // Domain leaderboard - multi-source aggregation
        $domainData = [];

        // Source 1: crawler_domains (real-time crawl stats)
        try {
            $domStmt = $pdo->query("SELECT domain, total_crawled as crawled, total_imported as imported, urls_errors as errors, quality_score as quality FROM crawler_domains WHERE total_crawled > 0 ORDER BY total_imported DESC LIMIT 50");
            $domainData = $domStmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {}

        // Source 2: If empty, try recent pinfeeds (last 24h for performance)
        if (empty($domainData)) {
            try {
                $domStmt = $pdo->query("SELECT source_domain as domain, COUNT(*) as imported, 0 as crawled, 0 as errors, 100 as quality FROM pinfeeds WHERE source_domain IS NOT NULL AND source_domain != '' AND pubDate > DATE_SUB(NOW(), INTERVAL 24 HOUR) GROUP BY source_domain ORDER BY imported DESC LIMIT 50");
                $domainData = $domStmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (Exception $e) {}
        }

        // Source 3: If still empty, get all-time top domains (cached approach)
        if (empty($domainData)) {
            try {
                $domStmt = $pdo->query("SELECT source_domain as domain, COUNT(*) as imported, 0 as crawled, 0 as errors, 100 as quality FROM pinfeeds WHERE source_domain IS NOT NULL AND source_domain != '' GROUP BY source_domain ORDER BY imported DESC LIMIT 50");
                $domainData = $domStmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (Exception $e) {}
        }
        $data['domain_leaderboard'] = $domainData;

        // Level stats (pyramid data) from active processes
        $levelStats = [];
        foreach ($procs as $p) {
            if (!empty($p['level_stats'])) {
                $ls = json_decode($p['level_stats'], true);
                if (is_array($ls)) {
                    foreach ($ls as $level => $count) {
                        $levelStats[$level] = ($levelStats[$level] ?? 0) + $count;
                    }
                }
            }
        }
        ksort($levelStats);
        $data['level_stats'] = $levelStats;

        // Network stats (edges count)
        try {
            $data['network'] = [
                'total_edges' => (int)$pdo->query("SELECT COUNT(*) FROM crawler_edges")->fetchColumn(),
                'total_nodes' => (int)$pdo->query("SELECT COUNT(DISTINCT domain) FROM crawler_domains")->fetchColumn()
            ];
        } catch (Exception $e) {
            $data['network'] = ['total_edges' => 0, 'total_nodes' => 0];
        }

        // Activity log (last 20 entries)
        $logStmt = $pdo->query("SELECT l.url, l.status, l.title, l.domain, l.created_at, p.mode FROM crawler_log l LEFT JOIN crawler_processes p ON l.process_id = p.id ORDER BY l.created_at DESC LIMIT 20");
        $data['activity_log'] = $logStmt->fetchAll(PDO::FETCH_ASSOC);

        // Alerts
        $alertStmt = $pdo->query("SELECT id, process_id, alert_type, message, created_at FROM crawler_alerts WHERE acknowledged = FALSE ORDER BY created_at DESC LIMIT 10");
        $data['alerts'] = $alertStmt->fetchAll(PDO::FETCH_ASSOC);

        // System health
        $data['system'] = [
            'memory_usage' => round(memory_get_usage(true) / 1024 / 1024, 2),
            'memory_peak' => round(memory_get_peak_usage(true) / 1024 / 1024, 2),
            'time' => date('Y-m-d H:i:s'),
            'uptime' => time()
        ];

        echo "data: " . json_encode($data) . "\n\n" . $padding;
        if (ob_get_level()) ob_flush();
        flush();

        sleep(2);
    }
    exit;
}

// ============================================
// DCI API Actions
// ============================================
if ($action === 'dci_pause_all') {
    header('Content-Type: application/json');
    $affected = $pdo->exec("UPDATE crawler_processes SET status='paused' WHERE status='running'");
    echo json_encode(['success' => true, 'affected' => $affected ?: 0]);
    exit;
}

if ($action === 'dci_resume_all') {
    header('Content-Type: application/json');
    $affected = $pdo->exec("UPDATE crawler_processes SET status='running' WHERE status='paused'");
    echo json_encode(['success' => true, 'affected' => $affected ?: 0]);
    exit;
}

if ($action === 'dci_stop_all') {
    header('Content-Type: application/json');
    $affected = $pdo->exec("UPDATE crawler_processes SET status='stopped', stopped_at=NOW() WHERE status IN ('running','paused','pending')");
    echo json_encode(['success' => true, 'affected' => $affected ?: 0]);
    exit;
}

if ($action === 'dci_cleanup_dead') {
    header('Content-Type: application/json');
    // Mark processes with old heartbeat as dead
    $affected = $pdo->exec("UPDATE crawler_processes SET status='dead', stopped_at=NOW() WHERE status='running' AND last_heartbeat < DATE_SUB(NOW(), INTERVAL 45 SECOND)");
    echo json_encode(['success' => true, 'affected' => $affected ?: 0]);
    exit;
}

if ($action === 'dci_control_process') {
    header('Content-Type: application/json');
    $processId = preg_replace('/[^a-f0-9]/', '', $_REQUEST['process_id'] ?? '');
    $ctrlAction = $_REQUEST['ctrl_action'] ?? '';

    if (!$processId || !in_array($ctrlAction, ['stop', 'pause', 'resume'])) {
        echo json_encode(['error' => 'Invalid parameters']);
        exit;
    }

    if ($ctrlAction === 'stop') {
        $pdo->prepare("UPDATE crawler_processes SET status='stopped', stopped_at=NOW() WHERE id=?")->execute([$processId]);
    } elseif ($ctrlAction === 'pause') {
        $pdo->prepare("UPDATE crawler_processes SET status='paused' WHERE id=? AND status='running'")->execute([$processId]);
    } elseif ($ctrlAction === 'resume') {
        $pdo->prepare("UPDATE crawler_processes SET status='running' WHERE id=? AND status='paused'")->execute([$processId]);
    }

    echo json_encode(['success' => true, 'action' => $ctrlAction, 'process_id' => $processId]);
    exit;
}

if ($action === 'dci_get_config') {
    header('Content-Type: application/json');
    $settings = [];
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM crawler_settings");
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    // Defaults
    $defaults = [
        'default_concurrency' => '10',
        'heartbeat_interval' => '15',
        'dead_timeout' => '45',
        'auto_cleanup' => '0',
        'max_processes' => '999999999',
        'default_wave_size' => '10'
    ];
    foreach ($defaults as $k => $v) {
        if (!isset($settings[$k])) $settings[$k] = $v;
    }
    echo json_encode(['success' => true, 'config' => $settings]);
    exit;
}

if ($action === 'dci_save_config') {
    header('Content-Type: application/json');
    $config = json_decode(file_get_contents('php://input'), true) ?: $_REQUEST;
    $allowed = ['default_concurrency', 'heartbeat_interval', 'dead_timeout', 'auto_cleanup', 'max_processes', 'default_wave_size'];

    $stmt = $pdo->prepare("INSERT INTO crawler_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    foreach ($allowed as $key) {
        if (isset($config[$key])) {
            $stmt->execute([$key, $config[$key]]);
        }
    }
    echo json_encode(['success' => true]);
    exit;
}

if ($action === 'dci_acknowledge_alert') {
    header('Content-Type: application/json');
    $alertId = (int)($_REQUEST['alert_id'] ?? 0);
    if ($alertId) {
        $pdo->prepare("UPDATE crawler_alerts SET acknowledged = TRUE WHERE id = ?")->execute([$alertId]);
    }
    echo json_encode(['success' => true]);
    exit;
}

if ($action === 'dci_clear_pool') {
    header('Content-Type: application/json');
    $status = $_REQUEST['status'] ?? 'done';
    if (in_array($status, ['done', 'pending', 'claimed', 'all'])) {
        if ($status === 'all') {
            $affected = $pdo->exec("DELETE FROM crawler_pool");
        } else {
            $stmt = $pdo->prepare("DELETE FROM crawler_pool WHERE status = ?");
            $stmt->execute([$status]);
            $affected = $stmt->rowCount();
        }
        echo json_encode(['success' => true, 'affected' => $affected]);
    } else {
        echo json_encode(['error' => 'Invalid status']);
    }
    exit;
}

if ($action === 'dci_clear_seen') {
    header('Content-Type: application/json');
    $affected = $pdo->exec("DELETE FROM crawler_seen");
    echo json_encode(['success' => true, 'affected' => $affected ?: 0]);
    exit;
}

if ($action === 'dci_get_process_details') {
    header('Content-Type: application/json');
    $processId = preg_replace('/[^a-f0-9]/', '', $_REQUEST['process_id'] ?? '');
    if (!$processId) {
        echo json_encode(['error' => 'Invalid process_id']);
        exit;
    }

    $stmt = $pdo->prepare("SELECT * FROM crawler_processes WHERE id = ?");
    $stmt->execute([$processId]);
    $process = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$process) {
        echo json_encode(['error' => 'Process not found']);
        exit;
    }

    // Get recent logs for this process
    $logStmt = $pdo->prepare("SELECT url, status, title, domain, created_at FROM crawler_log WHERE process_id = ? ORDER BY created_at DESC LIMIT 50");
    $logStmt->execute([$processId]);
    $process['logs'] = $logStmt->fetchAll(PDO::FETCH_ASSOC);

    // Get pool URLs for this process
    $poolStmt = $pdo->prepare("SELECT url, status, priority, level FROM crawler_pool WHERE process_id = ? ORDER BY priority DESC LIMIT 100");
    $poolStmt->execute([$processId]);
    $process['pool_urls'] = $poolStmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode(['success' => true, 'process' => $process]);
    exit;
}

// ============================================
// HTML PAGE
// ============================================
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>DCI Control Center v1.0</title>
    <link rel="stylesheet" href="/flowb0t.com/v2/public/assets/css/dci-control-center.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <div class="dci-container">
        <!-- Header -->
        <header class="dci-header">
            <div class="dci-logo">
                <span class="logo-icon">&#9881;</span>
                <h1>DCI Control Center <span class="version">v1.0</span></h1>
            </div>
            <div class="dci-nav">
                <a href="crawler-ultimate.php" class="nav-link">&#8592; Back to Crawler</a>
                <span class="connection-status" id="connectionStatus">
                    <span class="status-dot"></span>
                    <span class="status-text">Connecting...</span>
                </span>
            </div>
        </header>

        <!-- Global Stats Bar -->
        <div class="stats-bar">
            <div class="stat-card" data-stat="total">
                <div class="stat-value" id="statTotal">0</div>
                <div class="stat-label">Total Processes</div>
            </div>
            <div class="stat-card running" data-stat="running">
                <div class="stat-value" id="statRunning">0</div>
                <div class="stat-label">Running</div>
            </div>
            <div class="stat-card paused" data-stat="paused">
                <div class="stat-value" id="statPaused">0</div>
                <div class="stat-label">Paused</div>
            </div>
            <div class="stat-card" data-stat="pool">
                <div class="stat-value" id="statPool">0</div>
                <div class="stat-label">Pool Size</div>
            </div>
            <div class="stat-card success" data-stat="imports">
                <div class="stat-value" id="statImports">0</div>
                <div class="stat-label">Total Imports</div>
            </div>
            <div class="stat-card error" data-stat="errors">
                <div class="stat-value" id="statErrors">0</div>
                <div class="stat-label">Errors</div>
            </div>
            <div class="stat-card" data-stat="rate">
                <div class="stat-value" id="statRate">0</div>
                <div class="stat-label">URLs/min</div>
            </div>
        </div>

        <!-- Global Controls -->
        <div class="controls-bar">
            <h2>Global Controls</h2>
            <div class="control-buttons">
                <button class="btn btn-success" onclick="DCI.resumeAll()" title="Resume all paused processes">
                    <span class="btn-icon">&#9654;</span> Resume All
                </button>
                <button class="btn btn-warning" onclick="DCI.pauseAll()" title="Pause all running processes">
                    <span class="btn-icon">&#10074;&#10074;</span> Pause All
                </button>
                <button class="btn btn-danger" onclick="DCI.stopAll()" title="Stop all processes">
                    <span class="btn-icon">&#9632;</span> Stop All
                </button>
                <button class="btn btn-secondary" onclick="DCI.cleanupDead()" title="Remove dead processes">
                    <span class="btn-icon">&#128465;</span> Cleanup Dead
                </button>
                <button class="btn btn-info" onclick="DCI.showPoolManager()" title="Manage URL pool">
                    <span class="btn-icon">&#128200;</span> Pool Manager
                </button>
                <button class="btn btn-config" onclick="DCI.showConfig()" title="Configuration">
                    <span class="btn-icon">&#9881;</span> Config
                </button>
            </div>
        </div>

        <!-- Alerts Banner -->
        <div class="alerts-banner" id="alertsBanner" style="display: none;">
            <div class="alerts-content" id="alertsContent"></div>
        </div>

        <!-- Main Content Grid -->
        <div class="main-grid">
            <!-- Processes Table -->
            <div class="card processes-card">
                <div class="card-header">
                    <h3>Active Processes</h3>
                    <div class="card-actions">
                        <select id="modeFilter" onchange="DCI.filterProcesses()" title="Filter by mode">
                            <option value="all">All Modes</option>
                            <option value="viral">Viral</option>
                            <option value="ultimate">Ultimate</option>
                            <option value="multi">Multi</option>
                            <option value="infinite">Infinite</option>
                        </select>
                        <select id="statusFilter" onchange="DCI.filterProcesses()" title="Filter by status">
                            <option value="all">All Status</option>
                            <option value="running">Running</option>
                            <option value="paused">Paused</option>
                            <option value="pending">Pending</option>
                        </select>
                    </div>
                </div>
                <div class="card-body">
                    <table class="processes-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Mode</th>
                                <th>Status</th>
                                <th>Keywords</th>
                                <th>Pool</th>
                                <th>Imported</th>
                                <th>Errors</th>
                                <th>Rate</th>
                                <th>Elapsed</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="processesTable">
                            <tr class="loading-row">
                                <td colspan="10">Loading processes...</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Side Panel -->
            <div class="side-panel">
                <!-- Domain Leaderboard -->
                <div class="card leaderboard-card">
                    <div class="card-header">
                        <h3>Domain Leaderboard</h3>
                        <span class="badge" id="domainCount">0</span>
                    </div>
                    <div class="card-body">
                        <div class="leaderboard-list" id="leaderboardList">
                            <div class="loading">Loading domains...</div>
                        </div>
                    </div>
                </div>

                <!-- Activity Log -->
                <div class="card activity-card">
                    <div class="card-header">
                        <h3>Activity Log</h3>
                        <button class="btn-small" onclick="DCI.toggleAutoScroll()" title="Toggle auto-scroll">
                            <span id="autoScrollIcon">&#8595;</span>
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="activity-log" id="activityLog">
                            <div class="loading">Loading activity...</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts Row -->
        <div class="charts-row">
            <div class="card chart-card">
                <div class="card-header">
                    <h3>Mode Distribution</h3>
                </div>
                <div class="card-body">
                    <canvas id="modeChart"></canvas>
                </div>
            </div>
            <div class="card chart-card">
                <div class="card-header">
                    <h3>Import Velocity</h3>
                </div>
                <div class="card-body">
                    <canvas id="velocityChart"></canvas>
                </div>
            </div>
        </div>

        <!-- Pyramid & Network Row -->
        <div class="charts-row">
            <div class="card chart-card">
                <div class="card-header">
                    <h3>Pyramid Levels</h3>
                    <span class="badge" id="totalLevels">0 levels</span>
                </div>
                <div class="card-body">
                    <canvas id="pyramidChart"></canvas>
                </div>
            </div>
            <div class="card">
                <div class="card-header">
                    <h3>Network Stats</h3>
                </div>
                <div class="card-body">
                    <div class="network-stats-grid">
                        <div class="network-stat">
                            <div class="network-stat-value" id="networkNodes">0</div>
                            <div class="network-stat-label">Domains (Nodes)</div>
                        </div>
                        <div class="network-stat">
                            <div class="network-stat-value" id="networkEdges">0</div>
                            <div class="network-stat-label">Connections (Edges)</div>
                        </div>
                        <div class="network-stat">
                            <div class="network-stat-value" id="networkDensity">0%</div>
                            <div class="network-stat-label">Network Density</div>
                        </div>
                        <div class="network-stat">
                            <div class="network-stat-value" id="avgConnections">0</div>
                            <div class="network-stat-label">Avg Connections</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recently Completed -->
        <div class="card recent-card">
            <div class="card-header">
                <h3>Recently Completed/Stopped</h3>
            </div>
            <div class="card-body">
                <div class="recent-list" id="recentList">
                    <div class="loading">Loading recent processes...</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Config Modal -->
    <div class="modal" id="configModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Configuration</h3>
                <button class="modal-close" onclick="DCI.closeModal('configModal')">&times;</button>
            </div>
            <div class="modal-body">
                <div class="config-form">
                    <div class="form-group">
                        <label for="cfgConcurrency">Default Concurrency</label>
                        <input type="number" id="cfgConcurrency" min="1" max="100" value="10">
                    </div>
                    <div class="form-group">
                        <label for="cfgHeartbeat">Heartbeat Interval (seconds)</label>
                        <input type="number" id="cfgHeartbeat" min="5" max="60" value="15">
                    </div>
                    <div class="form-group">
                        <label for="cfgDeadTimeout">Dead Timeout (seconds)</label>
                        <input type="number" id="cfgDeadTimeout" min="30" max="300" value="45">
                    </div>
                    <div class="form-group">
                        <label for="cfgWaveSize">Default Wave Size</label>
                        <input type="number" id="cfgWaveSize" min="1" max="1000" value="10">
                    </div>
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" id="cfgAutoCleanup">
                            Auto-cleanup dead processes
                        </label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary" onclick="DCI.closeModal('configModal')">Cancel</button>
                <button class="btn btn-primary" onclick="DCI.saveConfig()">Save Configuration</button>
            </div>
        </div>
    </div>

    <!-- Pool Manager Modal -->
    <div class="modal" id="poolModal">
        <div class="modal-content modal-lg">
            <div class="modal-header">
                <h3>Pool Manager</h3>
                <button class="modal-close" onclick="DCI.closeModal('poolModal')">&times;</button>
            </div>
            <div class="modal-body">
                <div class="pool-stats">
                    <div class="pool-stat">
                        <span class="pool-stat-value" id="poolPending">0</span>
                        <span class="pool-stat-label">Pending</span>
                    </div>
                    <div class="pool-stat">
                        <span class="pool-stat-value" id="poolClaimed">0</span>
                        <span class="pool-stat-label">Claimed</span>
                    </div>
                    <div class="pool-stat">
                        <span class="pool-stat-value" id="poolDone">0</span>
                        <span class="pool-stat-label">Done</span>
                    </div>
                    <div class="pool-stat">
                        <span class="pool-stat-value" id="poolSeen">0</span>
                        <span class="pool-stat-label">Seen URLs</span>
                    </div>
                </div>
                <div class="pool-actions">
                    <button class="btn btn-warning" onclick="DCI.clearPool('done')">Clear Done URLs</button>
                    <button class="btn btn-danger" onclick="DCI.clearPool('all')">Clear All Pool</button>
                    <button class="btn btn-danger" onclick="DCI.clearSeen()">Clear Seen URLs</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Process Details Modal -->
    <div class="modal" id="processModal">
        <div class="modal-content modal-lg">
            <div class="modal-header">
                <h3>Process Details: <span id="processModalId"></span></h3>
                <button class="modal-close" onclick="DCI.closeModal('processModal')">&times;</button>
            </div>
            <div class="modal-body" id="processModalBody">
                Loading...
            </div>
        </div>
    </div>

    <script src="/flowb0t.com/v2/public/assets/js/dci-controller.js"></script>
    <script>
        // Initialize DCI Controller
        document.addEventListener('DOMContentLoaded', function() {
            DCI.init('<?php echo basename($_SERVER['PHP_SELF']); ?>?action=dci_sse');
        });
    </script>
</body>
</html>
