// Basic front-end logic: fetch feeds, render masonry cards, handle actions
const API = (action, params={}, method='GET', isForm=false) => {
  let url = `api/router.php?action=${encodeURIComponent(action)}`;
  const opts = { method, headers: {} };
  if (method === 'GET') {
    const q = new URLSearchParams(params).toString();
    if (q) url += '&' + q;
  } else {
    if (isForm) {
      opts.body = params;
    } else {
      opts.headers['Content-Type'] = 'application/x-www-form-urlencoded; charset=UTF-8';
      opts.body = new URLSearchParams(params).toString();
    }
  }
  return fetch(url, opts).then(r => r.json());
};

// Masonry rendering via CSS columns, just append cards
function cardTpl(item){
  const domain = item.source_domain || '';
  const a = item.author || 'Unknown';
  const d = item.pubDate ? new Date(item.pubDate.replace(' ', 'T')).toLocaleString() : '';
  const desc = (item.description || '').replace(/<[^>]+>/g, '').slice(0, 200);
  const img = item.thumbnail ? `<figure><img src="${item.thumbnail}" loading="lazy" alt=""></figure>` : '';
  const views = (item.views_7d || 0);
  const trend = item.trending_score != null ? item.trending_score.toFixed(2) : '—';
  return `
  <article class="card" data-id="${item.id}">
    ${img}
    <div class="content">
      <h3 class="title"><a href="view.php?id=${item.id}">${item.title}</a></h3>
      <div class="meta">
        <span class="kv"><span class="dot"></span> ${domain}</span>
        <span class="kv">By ${a}</span>
        <span class="kv" title="Published at">${d}</span>
        <span class="kv" title="7d views">${views} views</span>
        <span class="kv" title="Trending score">🔥 ${trend}</span>
      </div>
      ${desc ? `<p class="desc">${desc}</p>` : ''}
      <div class="actions">
        <button class="like" data-action="like">👍 <span class="l">${item.likes||0}</span></button>
        <button class="dislike" data-action="dislike">👎 <span class="d">${item.dislikes||0}</span></button>
        <button class="fav" data-action="favorite">⭐ Favorite</button>
        <a class="ghost" style="flex:1;text-align:center;padding:8px 10px;border-radius:10px;border:1px solid #2b3a60" href="${item.link}" target="_blank" rel="noopener">Open</a>
      </div>
    </div>
  </article>`;
}

async function loadFeeds(opts){
  const resp = await API('get_pinfeeds', opts, 'GET');
  if (!resp.ok) throw new Error(resp.message||resp.error||'Failed');
  const grid = document.querySelector('.grid');
  if (opts.page === 1) grid.innerHTML = '';
  resp.items.forEach(it => grid.insertAdjacentHTML('beforeend', cardTpl(it)));
  const moreBtn = document.getElementById('loadMore');
  moreBtn.dataset.page = String((opts.page||1) + 1);
  moreBtn.dataset.hasnext = resp.meta.has_next ? '1' : '0';
  moreBtn.style.display = resp.meta.has_next ? 'inline-block' : 'none';
}

function gatherFilters(){
  return {
    q: document.getElementById('q').value.trim(),
    domain: document.getElementById('domain').value.trim(),
    author: document.getElementById('author').value.trim(),
    tags: document.getElementById('tags').value.trim(),
    order: document.getElementById('order').value,
    per_page: document.getElementById('per_page').value,
  };
}

async function initHome(){
  document.getElementById('searchBtn').addEventListener('click', async () => {
    const filters = gatherFilters();
    filters.page = 1;
    try { await loadFeeds(filters); } catch(e){ alert(e.message); }
  });
  document.getElementById('loadMore').addEventListener('click', async (e)=>{
    const btn = e.currentTarget;
    if (btn.dataset.hasnext !== '1') return;
    const filters = gatherFilters();
    filters.page = parseInt(btn.dataset.page||'2', 10);
    try { await loadFeeds(filters); } catch(err){ alert(err.message); }
  });

  // card actions (like/dislike/favorite)
  document.addEventListener('click', async (ev) => {
    const btn = ev.target.closest('button');
    if (!btn) return;
    const action = btn.dataset.action;
    if (!action) return;
    const card = btn.closest('.card');
    if (!card) return;
    const id = card.dataset.id;
    try{
      if (action === 'favorite') {
        const resp = await API('favorite_toggle', {feed_id: id}, 'POST');
        if (!resp.ok) alert(resp.message||resp.error); else btn.classList.toggle('active');
      } else {
        const resp = await API('post_reaction', {feed_id: id, reaction: action}, 'POST');
        if (!resp.ok) { alert(resp.message||resp.error); return; }
        if (action === 'like'){
          const el = card.querySelector('.l'); el.textContent = String(parseInt(el.textContent||'0',10)+1);
          btn.classList.add('active');
          card.querySelector('.dislike').classList.remove('active');
        } else {
          const el = card.querySelector('.d'); el.textContent = String(parseInt(el.textContent||'0',10)+1);
          btn.classList.add('active');
          card.querySelector('.like').classList.remove('active');
        }
      }
    } catch(e){ alert(e.message); }
  });

  // initial load
  const initial = gatherFilters(); initial.page = 1;
  try { await loadFeeds(initial); } catch(e){ console.error(e); }
}

document.addEventListener('DOMContentLoaded', () => {
  if (document.body.dataset.page === 'home') initHome();
});
