<?php
require_once __DIR__ . '/../config.php';

/**
 * =========================================================
 *  File: api_comment_polls.php
 *  Purpose: Record a vote on a comment poll.
 *  Input (JSON, POST):
 *    { "poll_id": number, "option_id": number }
 *
 *  Behavior:
 *    - Requires logged-in user (session).
 *    - Ensures option belongs to poll.
 *    - Enforces 1 vote per user per poll (updates the selection if user changes).
 *    - Uses tables that match your schema:
 *        comments_polls, comments_poll_options, comments_poll_votes
 *
 *  Response:
 *    200: { "success": true }
 *    400/401/404/500: { "success": false, "error": "<code>" }
 *
 *  Notes:
 *    - Removed legacy CSRF header requirement causing 403s.
 *    - No schema changes required.
 * =========================================================
 */
session_start();

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

/* ---------- DB CONFIG ---------- */
$host    = 'localhost';
$db      = 'digupdog_FEED';
$user    = 'digupdog_FEEDadmin';
$pass    = 'Raimundinho1';
$charset = 'utf8mb4';

/* ---------- CONNECT ---------- */
try {
    $pdo = new PDO(
        "mysql:host={$host};dbname={$db};charset={$charset}",
        $user,
        $pass,
        [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success'=>false, 'error'=>'db_connection_failed']);
    exit;
}

/* ---------- AUTH ---------- */
if (empty($_SESSION['username'])) {
    http_response_code(401);
    echo json_encode(['success'=>false, 'error'=>'not_logged_in']);
    exit;
}

// Resolve current user (ID)
try {
    $st = $pdo->prepare("SELECT ID FROM user_myhashtag WHERE username = :u LIMIT 1");
    $st->execute([':u' => $_SESSION['username']]);
    $me = $st->fetch();
    if (!$me || empty($me['ID'])) {
        http_response_code(401);
        echo json_encode(['success'=>false, 'error'=>'not_logged_in']);
        exit;
    }
    $userId = (int)$me['ID'];
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success'=>false, 'error'=>'auth_lookup_failed']);
    exit;
}

/* ---------- INPUT ---------- */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success'=>false, 'error'=>'method_not_allowed']);
    exit;
}

$inRaw = file_get_contents('php://input');
$in    = json_decode($inRaw, true);

$pollId   = isset($in['poll_id'])   ? (int)$in['poll_id']   : 0;
$optionId = isset($in['option_id']) ? (int)$in['option_id'] : 0;

if ($pollId <= 0 || $optionId <= 0) {
    http_response_code(400);
    echo json_encode(['success'=>false, 'error'=>'invalid_params']);
    exit;
}

/* ---------- VALIDATION (poll + option) ---------- */
try {
    // Ensure poll exists
    $st = $pdo->prepare("SELECT poll_id FROM comments_polls WHERE poll_id = ? LIMIT 1");
    $st->execute([$pollId]);
    if (!$st->fetch()) {
        http_response_code(404);
        echo json_encode(['success'=>false, 'error'=>'poll_not_found']);
        exit;
    }

    // Ensure option belongs to this poll
    $st = $pdo->prepare("SELECT option_id FROM comments_poll_options WHERE option_id = ? AND poll_id = ? LIMIT 1");
    $st->execute([$optionId, $pollId]);
    if (!$st->fetch()) {
        http_response_code(404);
        echo json_encode(['success'=>false, 'error'=>'option_not_found']);
        exit;
    }
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success'=>false, 'error'=>'validation_failed']);
    exit;
}

/* ---------- VOTE (idempotent per user+poll) ---------- */
try {
    $pdo->beginTransaction();

    // Check existing vote from this user for this poll
    $st = $pdo->prepare("SELECT vote_id, option_id FROM comments_poll_votes WHERE poll_id = ? AND user_id = ? LIMIT 1");
    $st->execute([$pollId, $userId]);
    $existing = $st->fetch();

    if ($existing) {
        $voteId      = (int)$existing['vote_id'];
        $currentOpt  = (int)$existing['option_id'];

        if ($currentOpt !== $optionId) {
            // Update the user's choice
            $up = $pdo->prepare("UPDATE comments_poll_votes SET option_id = ?, created_at = NOW() WHERE vote_id = ?");
            $up->execute([$optionId, $voteId]);
        }
        // else: same option, nothing to change (idempotent)
    } else {
        // Insert a new vote
        $ins = $pdo->prepare("INSERT INTO comments_poll_votes (poll_id, option_id, user_id, created_at) VALUES (?, ?, ?, NOW())");
        $ins->execute([$pollId, $optionId, $userId]);
    }

    $pdo->commit();

    // Keep response small; UI re-pulls the poll with api_comments.php
    echo json_encode(['success'=>true]);
    exit;

} catch (Throwable $e) {
    if ($pdo->inTransaction()) { $pdo->rollBack(); }
    http_response_code(500);
    echo json_encode(['success'=>false, 'error'=>'vote_failed']);
    exit;
}
