<?php
require_once __DIR__ . '/../config.php';

/**
 * ====================================================================
 * File: api_comment_reactions.php (NEW)
 * 
 * Handles emoji reactions on comments (👍❤️😂🔥)
 * POST JSON { comment_id, reaction_type } → add/remove reaction
 * ====================================================================
 */

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_out(['success'=>false,'error'=>'Method not allowed'], 405);
}

$data = read_json();
$comment_id = isset($data['comment_id']) ? (int)$data['comment_id'] : 0;
$reaction_type = isset($data['reaction_type']) ? trim((string)$data['reaction_type']) : '';

if ($comment_id <= 0) {
    json_out(['success'=>false,'error'=>'comment_id required'], 400);
}

// Validate reaction type
$valid_reactions = ['thumbs_up', 'heart', 'laugh', 'fire'];
if (!in_array($reaction_type, $valid_reactions)) {
    json_out(['success'=>false,'error'=>'Invalid reaction type'], 400);
}

// Rate limiting
$reaction_rate_key = get_client_ip() . ':reactions';
if (!check_rate_limit($reaction_rate_key, 30, 60)) { // 30 reactions per minute
    json_out(['success'=>false,'error'=>'Too many reactions'], 429);
}

// Verify comment exists
$comment_check = $conn->prepare("SELECT comment_id FROM comments WHERE comment_id=? AND is_deleted=0 LIMIT 1");
$comment_check->bind_param('i', $comment_id);
$comment_check->execute();

if (!$comment_check->get_result()->fetch_assoc()) {
    json_out(['success'=>false,'error'=>'Comment not found'], 404);
}

$voter_key = voter_key();
$username = current_username();

// Check existing reaction
$existing_check = $conn->prepare("SELECT id, reaction_type FROM comment_reactions WHERE comment_id=? AND voter_key=? LIMIT 1");
$existing_check->bind_param('is', $comment_id, $voter_key);
$existing_check->execute();
$existing = $existing_check->get_result()->fetch_assoc();

$conn->begin_transaction();

try {
    if ($existing) {
        if ($existing['reaction_type'] === $reaction_type) {
            // Remove existing reaction (toggle off)
            $remove_stmt = $conn->prepare("DELETE FROM comment_reactions WHERE id=?");
            $remove_stmt->bind_param('i', $existing['id']);
            $remove_stmt->execute();
            $action = 'removed';
        } else {
            // Change reaction type
            $update_stmt = $conn->prepare("UPDATE comment_reactions SET reaction_type=?, created_at=NOW() WHERE id=?");
            $update_stmt->bind_param('si', $reaction_type, $existing['id']);
            $update_stmt->execute();
            $action = 'changed';
        }
    } else {
        // Add new reaction
        $insert_stmt = $conn->prepare("INSERT INTO comment_reactions (comment_id, reaction_type, voter_key, username, ip_address, created_at) VALUES (?,?,?,?,?,NOW())");
        $ip = get_client_ip();
        $insert_stmt->bind_param('issss', $comment_id, $reaction_type, $voter_key, $username, $ip);
        $insert_stmt->execute();
        $action = 'added';
    }
    
    $conn->commit();
    
    // Get updated reaction counts
    $reactions = get_comment_reactions($conn, $comment_id);
    $user_reaction = $action === 'removed' ? null : $reaction_type;
    
    json_out([
        'success' => true,
        'action' => $action,
        'comment_id' => $comment_id,
        'reaction_type' => $reaction_type,
        'user_reaction' => $user_reaction,
        'reactions' => $reactions
    ]);
    
} catch (Exception $e) {
    $conn->rollback();
    error_log("Reaction error: " . $e->getMessage());
    json_out(['success'=>false,'error'=>'Reaction failed'], 500);
}

function get_comment_reactions(mysqli $conn, int $comment_id): array {
    $stmt = $conn->prepare("
        SELECT reaction_type, COUNT(*) as count 
        FROM comment_reactions 
        WHERE comment_id=? 
        GROUP BY reaction_type
    ");
    $stmt->bind_param('i', $comment_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $reactions = ['thumbs_up' => 0, 'heart' => 0, 'laugh' => 0, 'fire' => 0];
    
    while ($row = $result->fetch_assoc()) {
        $reactions[$row['reaction_type']] = (int)$row['count'];
    }
    
    return $reactions;
}
?>