/**
 * Feed Rendering Module
 */

const Feed = {
  currentPage: 1,
  isLoading: false,
  hasMore: true,

  /**
   * Render posts to container
   */
  async renderPosts(posts, containerSelector = '#feed-container') {
    const container = document.querySelector(containerSelector);
    if (!container) return;

    posts.forEach(post => {
      const postElement = this.createPostCard(post);
      container.appendChild(postElement);
    });

    // Initialize tags and masonry
    if (typeof initializeTags === 'function') {
      initializeTags();
    }

    if (typeof initMasonry === 'function') {
      initMasonry();
    }
  },

  /**
   * Create post card HTML
   */
  createPostCard(post) {
    const card = document.createElement('div');
    card.className = 'post-card masonry-item';
    card.setAttribute('data-id', post.id);

    card.innerHTML = `
      <div class="post-header">
        <div class="post-author">
          <img src="${post.author_avatar || 'assets/images/default-avatar.png'}"
               alt="${post.author_name}"
               class="avatar">
          <div>
            <h4 class="author-name">${post.author_name}</h4>
            <span class="post-time">${this.formatTime(post.created_at)}</span>
          </div>
        </div>
        <button class="btn-more">⋯</button>
      </div>

      <div class="post-content">
        <h3 class="post-title">${post.title || ''}</h3>
        <p class="post-text">${post.text || ''}</p>

        ${post.image ? `<img src="${post.image}" class="post-image" alt="">` : ''}

        <div class="post-tags-container">
          <div class="post-tags"></div>
        </div>
      </div>

      <div class="m-actions">
        <button class="btn-action" onclick="Feed.handleLike(${post.id})">
          ❤️ <span>${post.likes || 0}</span>
        </button>
        <button class="btn-action" onclick="Feed.handleComment(${post.id})">
          💬 <span>${post.comments || 0}</span>
        </button>
        <button class="btn-action" onclick="Feed.handleShare(${post.id})">
          🔗 <span>${post.shares || 0}</span>
        </button>
        <button class="btn-action" onclick="Feed.handleSave(${post.id})">
          🔖
        </button>
      </div>
    `;

    return card;
  },

  /**
   * Format timestamp
   */
  formatTime(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = Math.floor((now - date) / 1000);

    if (diff < 60) return 'Just now';
    if (diff < 3600) return `${Math.floor(diff / 60)}m ago`;
    if (diff < 86400) return `${Math.floor(diff / 3600)}h ago`;
    return `${Math.floor(diff / 86400)}d ago`;
  },

  /**
   * Load more posts
   */
  async loadMore() {
    if (this.isLoading || !this.hasMore) return;

    this.isLoading = true;

    try {
      const response = await API.getPosts(this.currentPage + 1);

      if (response.success && response.posts.length > 0) {
        await this.renderPosts(response.posts);
        this.currentPage++;
        this.hasMore = response.has_more;
      } else {
        this.hasMore = false;
      }
    } catch (error) {
      console.error('Failed to load posts:', error);
      UI.showToast('Failed to load posts', 'error');
    } finally {
      this.isLoading = false;
    }
  },

  /**
   * Action handlers
   */
  async handleLike(postId) {
    try {
      const response = await API.likePost(postId);
      if (response.success) {
        const btn = document.querySelector(`.post-card[data-id="${postId}"] .btn-action:nth-child(1) span`);
        if (btn) btn.textContent = response.likes;
      }
    } catch (error) {
      console.error('Like failed:', error);
    }
  },

  async handleComment(postId) {
    UI.openModal('comment-modal');
    // Set post ID for modal
    document.getElementById('comment-modal').setAttribute('data-post-id', postId);
  },

  async handleShare(postId) {
    try {
      const response = await API.sharePost(postId);
      if (response.success) {
        UI.showToast('Post shared!', 'success');
      }
    } catch (error) {
      console.error('Share failed:', error);
    }
  },

  async handleSave(postId) {
    try {
      const response = await API.favoritePost(postId);
      if (response.success) {
        UI.showToast('Post saved!', 'success');
      }
    } catch (error) {
      console.error('Save failed:', error);
    }
  }
};

// Infinite scroll
let scrollTimeout;
window.addEventListener('scroll', () => {
  clearTimeout(scrollTimeout);
  scrollTimeout = setTimeout(() => {
    const scrolled = window.innerHeight + window.scrollY;
    const threshold = document.documentElement.scrollHeight - 1000;

    if (scrolled >= threshold) {
      Feed.loadMore();
    }
  }, 100);
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
  module.exports = Feed;
}
