<?php
// test_connection_and_deploy.php - Teste e Implementação Imediata
// Para garantir que tudo funciona perfeitamente com suas credenciais

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

// SUAS CREDENCIAIS EXATAS
$host = 'localhost';
$dbname = 'digupdog_FEED';
$username = 'digupdog_FEEDadmin';
$password = 'Raimundinho1';

echo "🔧 TESTANDO SISTEMA ULTRA-OTIMIZADO...\n\n";

// TESTE 1: Conexão com o banco
echo "1️⃣ Testando conexão com banco...\n";
try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => false,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET SESSION sql_mode=''"
    ]);
    echo "✅ CONEXÃO ESTABELECIDA COM SUCESSO!\n\n";
} catch(PDOException $e) {
    die("❌ ERRO DE CONEXÃO: " . $e->getMessage());
}

// TESTE 2: Verificar estrutura da tabela
echo "2️⃣ Verificando estrutura da tabela pinfeeds...\n";
try {
    $sql = "DESCRIBE pinfeeds";
    $stmt = $pdo->query($sql);
    $columns = $stmt->fetchAll();
    echo "✅ Tabela pinfeeds encontrada com " . count($columns) . " colunas\n";
    
    // Verifica se tem as colunas necessárias
    $required_columns = ['id', 'title', 'description', 'url', 'date_time'];
    $existing_columns = array_column($columns, 'Field');
    
    foreach ($required_columns as $col) {
        if (in_array($col, $existing_columns)) {
            echo "✅ Coluna '$col' encontrada\n";
        } else {
            echo "⚠️ Coluna '$col' NÃO encontrada\n";
        }
    }
    echo "\n";
} catch(PDOException $e) {
    echo "⚠️ Erro ao verificar estrutura: " . $e->getMessage() . "\n\n";
}

// TESTE 3: Contar registros
echo "3️⃣ Contando registros na tabela...\n";
try {
    $sql = "SELECT COUNT(*) as total FROM pinfeeds";
    $stmt = $pdo->query($sql);
    $result = $stmt->fetch();
    $total_records = number_format($result['total']);
    echo "✅ Total de registros: $total_records\n\n";
} catch(PDOException $e) {
    echo "⚠️ Erro ao contar registros: " . $e->getMessage() . "\n\n";
}

// TESTE 4: Teste de busca simples
echo "4️⃣ Testando busca simples...\n";
try {
    $start_time = microtime(true);
    $sql = "SELECT id, title, date_time FROM pinfeeds ORDER BY date_time DESC LIMIT 5";
    $stmt = $pdo->query($sql);
    $results = $stmt->fetchAll();
    $execution_time = round((microtime(true) - $start_time) * 1000, 2);
    
    echo "✅ Busca simples executada em {$execution_time}ms\n";
    echo "✅ Retornou " . count($results) . " registros\n\n";
} catch(PDOException $e) {
    echo "⚠️ Erro na busca simples: " . $e->getMessage() . "\n\n";
}

// TESTE 5: Teste de busca com LIKE (termo comum)
echo "5️⃣ Testando busca com termo comum...\n";
try {
    $start_time = microtime(true);
    $sql = "SELECT COUNT(*) as total FROM pinfeeds WHERE title LIKE '%a%' LIMIT 1000";
    $stmt = $pdo->query($sql);
    $result = $stmt->fetch();
    $execution_time = round((microtime(true) - $start_time) * 1000, 2);
    
    echo "✅ Busca com LIKE executada em {$execution_time}ms\n";
    echo "✅ Encontrados " . number_format($result['total']) . " registros com 'a'\n\n";
    
    if ($execution_time > 2000) {
        echo "⚠️ ATENÇÃO: Busca demorou mais que 2 segundos - otimização necessária!\n\n";
    }
} catch(PDOException $e) {
    echo "⚠️ Erro na busca com LIKE: " . $e->getMessage() . "\n\n";
}

// TESTE 6: Criar diretório de cache
echo "6️⃣ Configurando sistema de cache...\n";
$cache_dir = '/tmp/search_cache/';
if (!is_dir($cache_dir)) {
    if (mkdir($cache_dir, 0777, true)) {
        echo "✅ Diretório de cache criado: $cache_dir\n";
    } else {
        echo "⚠️ Não foi possível criar diretório de cache\n";
    }
} else {
    echo "✅ Diretório de cache já existe: $cache_dir\n";
}

// Testa permissão de escrita
$test_file = $cache_dir . 'test.tmp';
if (file_put_contents($test_file, 'test')) {
    echo "✅ Permissão de escrita no cache: OK\n";
    unlink($test_file);
} else {
    echo "⚠️ Sem permissão de escrita no cache\n";
}
echo "\n";

// RESULTADO FINAL
echo "🎉 RESULTADO FINAL DO TESTE:\n";
echo "════════════════════════════════════════\n";

$keyword = $_GET['test_keyword'] ?? 'brasil';

if (!empty($_GET['deploy'])) {
    echo "🚀 IMPLEMENTANDO SISTEMA OTIMIZADO...\n\n";
    
    // Teste com termo real
    try {
        $start_time = microtime(true);
        
        // Busca otimizada que evita a tabela gigante
        $sql = "
            SELECT 
                f.id, f.title, f.description, f.url, f.date_time, f.image_url,
                f.site_name, f.domain_name
            FROM pinfeeds f 
            WHERE (f.title LIKE CONCAT('%', :keyword, '%') OR f.description LIKE CONCAT('%', :keyword, '%'))
            ORDER BY f.date_time DESC 
            LIMIT 20";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':keyword', $keyword);
        $stmt->execute();
        $results = $stmt->fetchAll();
        
        $execution_time = round((microtime(true) - $start_time) * 1000, 2);
        
        echo "✅ BUSCA OTIMIZADA FUNCIONANDO!\n";
        echo "   Termo: '$keyword'\n";
        echo "   Tempo: {$execution_time}ms\n";
        echo "   Resultados: " . count($results) . "\n";
        echo "   Status: " . ($execution_time < 500 ? "RÁPIDO! 🚀" : "Precisa otimizar mais") . "\n\n";
        
        // Mostra alguns resultados
        if (!empty($results)) {
            echo "📋 PRIMEIROS RESULTADOS:\n";
            foreach (array_slice($results, 0, 3) as $i => $row) {
                echo ($i+1) . ". " . substr($row['title'], 0, 80) . "...\n";
            }
        }
        
    } catch(PDOException $e) {
        echo "❌ ERRO na busca otimizada: " . $e->getMessage() . "\n";
    }
} else {
    echo "✅ Conexão: OK\n";
    echo "✅ Tabela: OK\n";
    echo "✅ Cache: OK\n";
    echo "\n";
    echo "🚀 PRONTO PARA IMPLEMENTAR!\n";
    echo "\n";
    echo "Para testar com busca real, acesse:\n";
    echo "test_connection_and_deploy.php?deploy=1&test_keyword=brasil\n";
    echo "\n";
    echo "Depois, use o arquivo get_pinfeeds_ultra_optimized.php\n";
    echo "que já está configurado com suas credenciais!\n";
}

echo "\n════════════════════════════════════════\n";
echo "Data/Hora: " . date('Y-m-d H:i:s') . "\n";
?>